import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { BackgroundBannerContainer, mapStateToProps,
         createBGImageUrl } from 'ui/containers/background-banner';
import { BackgroundBanner as BackgroundBannerComponent } from 'ui/components/background-banner';
import { CONTENT_MODE_LIVE } from 'stream/twitch-live';
import { OFFLINE_STATUS, ONLINE_STATUS } from 'state/online-status';
import { CONTENT_MODE_NONE } from 'stream/null';
import { AdContentTypes } from 'actions/ads';

const NONE = 'none';
const OFFLINE_BANNER = 'offline-banner';
const VIDEO_PREVIEW = 'video-preview';

const DEFAULT_ARGS = Object.freeze({
    bannerType: NONE,
    imgUrl: '',
    onContentShowing() {},
});

const TEST_URLS = Object.freeze([
    'https://static-cdn.jtvnw.net/previews-ttv/live_user_monstercat-{width}x{height}.jpg',
    'https://static-cdn.jtvnw.net/previews-ttv/live_user_monstercat-320x240.jpg',
]);

const EXPECTED_IMG_URLS = Object.freeze({
    [NONE]: '',
    [OFFLINE_BANNER]: 'https://memes.org/banner.live_user-1280x720.gif',
    [VIDEO_PREVIEW]: 'https://my.memes.org/meme.live_user-1280x720.gif',
});

const FAKE_STATE = Object.freeze({
    streamMetadata: {
        channel: {
            videoBanner: 'https://memes.org/banner.live_user-320x240.gif',
        },
        preview: {
            template: 'https://my.memes.org/meme.live_user-{width}x{height}.gif',
        },
    },
    playback: {
        paused: false,
        autoplay: true,
        ended: false,
        hasPlayed: false,
    },
    stream: {
        contentType: CONTENT_MODE_LIVE,
    },
    onlineStatus: OFFLINE_STATUS,
    ads: {
        currentMetadata: {
            contentType: AdContentTypes.NONE,
        },
    },
});

function generateState(overrides = {}) {
    return assign({}, FAKE_STATE, overrides);
}

function renderBackgroundBannerContainer(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <BackgroundBannerContainer {...args} />;
    return shallow(component);
}

reactTest('ui | containers | background-banner', function() {
    QUnit.test('createBGImageUrl', function(assert) {
        TEST_URLS.forEach(url => {
            const expectedString = 'https://static-cdn.jtvnw.net/previews-ttv/live_user_monstercat-1280x720.jpg';

            assert.equal(
                createBGImageUrl(url),
                expectedString,
                'replaces resolution'
            );
        });
    });

    QUnit.test('renders null if bannerType is none', function(assert) {
        const component = renderBackgroundBannerContainer();
        assert.equal(component.type(), null, 'renders nothing');
    });

    QUnit.test('renders a BackgroundBanner Component if valid bannerType', function(assert) {
        const component = renderBackgroundBannerContainer({
            bannerType: OFFLINE_BANNER,
        });
        assert.equal(component.type(), BackgroundBannerComponent);
    });

    QUnit.test('invokes onContentShowing when props.bannerType is not NONE', function(assert) {
        const onContentShowing = sinon.spy();

        const component = renderBackgroundBannerContainer({
            onContentShowing,
        });

        component.setProps({ bannerType: VIDEO_PREVIEW });
        assert.equal(onContentShowing.callCount, 1, 'onContentShowing called once');
    });

    QUnit.test('passes imgUrl to component', function(assert) {
        const imgUrl = QUnit.config.current.testId;
        const component = renderBackgroundBannerContainer({
            imgUrl,
            bannerType: OFFLINE_BANNER,
        });

        assert.equal(component.props().imgUrl, imgUrl, 'imgUrl is passed to component');
    });

    QUnit.module('mapStateToProps', function() {
        function testBannerType(state, bannerType, msg) {
            QUnit.test(`should render ${bannerType} banner type if ${msg}`, function(assert) {
                const props = mapStateToProps(generateState(state));
                const expectedImgUrl = EXPECTED_IMG_URLS[bannerType];
                assert.equal(props.bannerType, bannerType, `should render ${bannerType}`);
                assert.equal(props.imgUrl, expectedImgUrl, 'renders correct image');
            });
        }

        testBannerType({
            ads: {
                currentMetadata: {
                    contentType: AdContentTypes.PREROLL,
                },
            },
        }, NONE, 'ad is playing');

        testBannerType({
            onlineStatus: ONLINE_STATUS,
            playback: {
                paused: false,
                autoplay: false,
                hasPlayed: true,
                ended: false,
            },
        }, NONE, 'online and hasPlayed, but hasn\'t ended');

        testBannerType({
            playback: {
                ended: true,
                autoplay: false,
                hasPlayed: true,
                paused: true,
            },
            onlineStatus: OFFLINE_STATUS,
        }, OFFLINE_BANNER, 'live, no ads playing, offline, and ended');

        testBannerType({
            playback: {
                ended: false,
                autoplay: false,
                hasPlayed: false,
                paused: true,
            },
            onlineStatus: ONLINE_STATUS,
        }, VIDEO_PREVIEW, 'autoplay is false, and content hasn\'t started');

        testBannerType({
            stream: {
                contentType: CONTENT_MODE_NONE,
            },
        }, VIDEO_PREVIEW, 'no content mode set');
    });
});
