import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { ChromecastButtonContainer } from 'ui/containers/buttons/chromecast-button';
import { AVAILABLE, CONNECTING, CONNECTED } from 'state/chromecast';
import { PLAYER_HIGHLIGHTER } from 'util/player-type';
import {
    ChromecastButton as ChromecastButtonComponent,
} from 'ui/components/buttons/chromecast-button';

const DEFAULT_ARGS = Object.freeze({
    castingState: '',
    playerType: '',
    startCast() {},
    stopCast() {},
});

function renderChromecastButton(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <ChromecastButtonContainer {...args} />;
    return shallow(component);
}

reactTest('ui | containers | buttons | chromecast-button', function() {
    QUnit.test('Renders a ChromecastButton component', function(assert) {
        const component = renderChromecastButton();
        assert.equal(component.type(), ChromecastButtonComponent);
    });

    QUnit.test('renders nothing if player type is highlighter', function(assert) {
        const component = renderChromecastButton({
            playerType: PLAYER_HIGHLIGHTER,
        });

        assert.equal(component.type(), null);
    });

    QUnit.test('starts cast on click when castingState is available', function(assert) {
        const startCast = sinon.spy();
        const stopCast = sinon.spy();
        const component = renderChromecastButton({
            castingState: AVAILABLE,
            startCast,
            stopCast,
        });

        component.simulate('click');

        assert.equal(startCast.callCount, 1, 'startCast called once');
        assert.equal(stopCast.callCount, 0, 'stopCast not called');
    });

    QUnit.test('stops cast on click when castingState is connected', function(assert) {
        const startCast = sinon.spy();
        const stopCast = sinon.spy();
        const component = renderChromecastButton({
            castingState: CONNECTED,
            startCast,
            stopCast,
        });

        component.simulate('click');

        assert.equal(stopCast.callCount, 1, 'stopCast called once');
        assert.equal(startCast.callCount, 0, 'startCast not called');
    });

    QUnit.test('NOOP on click when castingState is not connected or available', function(assert) {
        const startCast = sinon.spy();
        const stopCast = sinon.spy();
        const component = renderChromecastButton({
            castingState: CONNECTING,
            startCast,
            stopCast,
        });

        component.simulate('click');

        assert.equal(stopCast.callCount, 0, 'stopCast not called');
        assert.equal(startCast.callCount, 0, 'startCast not called');
    });
});
