import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { ClipsButtonContainer, mapDispatchToProps, mapStateToProps } from 'ui/containers/buttons/clips-button';
import { ClipsButton } from 'ui/components/buttons/clips-button';
import { trackEvent } from 'actions/analytics-tracker';
import { init as initStore } from 'state';
import { getValidClippingState } from 'tests/src/js/state/clips.test.js';

const DEFAULT_ARGS = Object.freeze({
    clipGenerator: {
        recordClip() {},
    },
    isSitePlayer: true,
    trackClipRecord() {},
    userCanClip: true,
});

function renderContainer(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const container = <ClipsButtonContainer {...args} />;
    return shallow(container);
}

reactTest('ui | containers | buttons | clips-button', function() {
    function testProp(propName, value, shouldRender) {
        const description = shouldRender
            ? `if ${propName} is ${value}, renders a ClipsButton`
            : `if ${propName} is ${value}, does not render anything`;

        QUnit.test(description, function(assert) {
            const container = renderContainer({
                [propName]: value,
            });

            assert.equal(container.type(), shouldRender ? ClipsButton : null);
        });
    }

    testProp('userCanClip', true, true);
    testProp('userCanClip', false, false);

    QUnit.test('the onClip prop invokes recordClip on the generator', function(assert) {
        const fakeGenerator = {
            recordClip: sinon.spy(),
        };
        const container = renderContainer({
            clipGenerator: fakeGenerator,
        });

        container.find(ClipsButton).props().onClip();

        assert.equal(fakeGenerator.recordClip.callCount, 1);
    });

    QUnit.test('hasAltShortcut prop is true if site player', function(assert) {
        const container = renderContainer({
            isSitePlayer: true,
        });

        assert.equal(container.find(ClipsButton).props().hasAltShortcut, true);
    });

    QUnit.module('mapDispatchToProps', function() {
        QUnit.test('trackClipRecord dispatches expected action', function(assert) {
            const dispatchSpy = sinon.spy();
            const mappedDispatches = mapDispatchToProps(dispatchSpy);
            mappedDispatches.trackClipRecord();

            assert.equal(dispatchSpy.callCount, 1, 'one action dispatched');
            assert.ok(dispatchSpy.calledWith(trackEvent('player_click_clip')), 'track event dispatched');
        });
    });

    QUnit.module('mapStateToProps', function() {
        function testState(propName, value, expectedState, additionalDescription = 'expected') {
            QUnit.test(`${propName} is ${value} when ${additionalDescription}`, function(assert) {
                const state = assign({}, initStore().getState(), expectedState);
                const mappedState = mapStateToProps(state);
                assert.equal(mappedState[propName], value);
            });
        }

        testState('userCanClip', true, getValidClippingState());
        testState('userCanClip', false, assign({}, getValidClippingState(), {
            user: {
                loggedInStatus: false,
            },
        }));
    });
});
