import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { ClosedCaptionButtonContainer, mapDispatchToProps } from 'ui/containers/buttons/closed-caption-button';
import { ClosedCaptionButton } from 'ui/components/buttons/closed-caption-button';
import { setCaptionsEnabled } from 'actions/captions';
import { trackEvent } from 'actions/analytics-tracker';

const DEFAULT_ARGS = Object.freeze({
    captionsAvailable: true,
    captionsEnabled: false,
    onEnableCaptions() {},
    onDisableCaptions() {},
});

function renderContainer(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const container = <ClosedCaptionButtonContainer {...args} />;
    return shallow(container);
}

reactTest('ui | containers | buttons | closed-caption-button', function() {
    QUnit.test('renders a ClosedCaptionButton', function(assert) {
        const container = renderContainer();
        assert.equal(container.type(), ClosedCaptionButton);
    });

    QUnit.test('renders nothing if captions are not available', function(assert) {
        const container = renderContainer({
            captionsAvailable: false,
        });
        assert.equal(container.type(), null);
    });

    QUnit.test('passes in expected props', function(assert) {
        const onEnableCaptions = sinon.spy();
        const onDisableCaptions = sinon.spy();
        const container = renderContainer({
            captionsEnabled: true,
            onEnableCaptions,
            onDisableCaptions,
        });

        container.find(ClosedCaptionButton).props().onEnableCaptions();

        assert.equal(onEnableCaptions.callCount, 1, 'onEnableCaptions prop invoked');
        assert.equal(onDisableCaptions.callCount, 0, 'onDisableCaptions prop not invoked');

        container.find(ClosedCaptionButton).props().onDisableCaptions();

        assert.equal(onEnableCaptions.callCount, 1, 'onEnableCaptions not invoked twice');
        assert.equal(onDisableCaptions.callCount, 1, 'onDisableCaptions invoked');

        assert.equal(
            container.find(ClosedCaptionButton).props().captionsEnabled,
            true,
            'captionsEnabled prop is passed in'
        );
    });

    QUnit.test('mapDispatchToProps dispatches expected actions', function(assert) {
        const dispatch = sinon.spy();
        const getState = () => {
            return {
                captions: {},
            };
        };

        const mappedDispatches = mapDispatchToProps(dispatch);

        mappedDispatches.onEnableCaptions();
        assert.equal(dispatch.callCount, 2, 'two actions dispatched');

        let firstDispatchAction = dispatch.firstCall.args[0](dispatch, getState);
        let expectedDispatchAction = setCaptionsEnabled(true)(dispatch, getState);

        assert.equal(firstDispatchAction, expectedDispatchAction, 'enable captions action dispatched');
        assert.deepEqual(dispatch.secondCall.args[0], trackEvent('player_click_cc'), 'track event action dispatched');

        dispatch.reset();

        mappedDispatches.onDisableCaptions();
        assert.equal(dispatch.callCount, 2, 'two actions dispatched');

        firstDispatchAction = dispatch.firstCall.args[0](dispatch, getState);
        expectedDispatchAction = setCaptionsEnabled(false)(dispatch, getState);

        assert.equal(firstDispatchAction, expectedDispatchAction, 'disable captions action dispatched');
        assert.deepEqual(dispatch.secondCall.args[0], trackEvent('player_click_cc'), 'track event action dispatched');
    });
});

