import { reactTest } from 'tests/utils/react-test';
import sinon from 'sinon';
import { FollowButtonComponent, mapStateToProps } from 'ui/containers/buttons/follow-button';
import * as buttonTestHelper from 'tests/utils/player-buttons-helpers';
import { mockTranslateFunc } from 'tests/utils/translate-hoc-helpers';

const DEFAULT_ARGS = Object.freeze({
    channelName: 'monstercat',
    isOwnChannel: false,
    isFollowing: false,
    onFollowChannel: () => {},
    onUnfollowChannel: () => {},
    t: mockTranslateFunc,
});

reactTest('ui | components | buttons', function() {
    QUnit.module('follow button', function(hooks) {
        hooks.beforeEach(function() {
            this.renderFollowButton = buttonTestHelper.createRenderFn(
                FollowButtonComponent,
                DEFAULT_ARGS
            );
        });

        QUnit.test('mapStateToProps computes isOwnChannel', function(assert) {
            const mockedState = {
                streamMetadata: {
                    channel: {
                        id: 123,
                    },
                },
                user: {
                    id: 123,
                },
                follow: {},
            };

            const mockedProps = mapStateToProps(mockedState);
            assert.equal(mockedProps.isOwnChannel, true, 'isOwnChannel is truthy');

            const user = { id: 999 };
            const notOwnerState = Object.assign({}, mockedState, { user });
            const notOwnerProps = mapStateToProps(notOwnerState);
            assert.equal(notOwnerProps.isOwnChannel, false, 'isOwnChannel is falsey');
        });

        QUnit.test('When following', function(assert) {
            const onFollowChannel = sinon.spy();
            const onUnfollowChannel = sinon.spy();
            const component = this.renderFollowButton({
                isFollowing: true,
                onFollowChannel,
                onUnfollowChannel,
            });

            const hoverSpan = component.childAt(0);
            const primarySpan = component.childAt(1);

            assert.equal(
                primarySpan.text(),
                'Following',
                'Shows user is following'
            );

            assert.ok(primarySpan.hasClass('pl-button__status--primary'));

            assert.equal(
                hoverSpan.text(),
                'Unfollow',
                'Shows user is following'
            );

            assert.ok(hoverSpan.hasClass('pl-button__status--hover'));

            component.simulate('click');

            assert.equal(onUnfollowChannel.callCount, 1, 'unfollow is called');
            assert.equal(onFollowChannel.callCount, 0, 'follow is not called');
        });

        QUnit.test('When not following', function(assert) {
            const onFollowChannel = sinon.spy();
            const onUnfollowChannel = sinon.spy();
            const channelName = QUnit.config.current.testId;

            const component = this.renderFollowButton({
                following: false,
                onFollowChannel,
                onUnfollowChannel,
                channelName,
            });

            assert.equal(
                component.childAt(1).text(),
                `Follow ${channelName} on Twitch`,
                'Asks user to follow the channel.'
            );

            component.simulate('click');

            assert.equal(onFollowChannel.callCount, 1, 'follow is called');
            assert.equal(onUnfollowChannel.callCount, 0, 'unfollow is not called');
        });
    });
});
