import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { FullScreenButtonContainer, mapDispatchToProps,
         mapStateToProps } from 'ui/containers/buttons/fullscreen-button';
import { FullScreenButton } from 'ui/components/buttons/fullscreen-button';
import { setFullScreen } from 'actions/screen-mode';
import { trackEvent } from 'actions/analytics-tracker';
import { init as initStore } from 'state';

const DEFAULT_ARGS = Object.freeze({
    canFullScreen: true,
    isBrandingShown: false,
    isFullScreen: false,
    onFullScreen() {},
    onUnFullScreen() {},
});

function renderContainer(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const container = <FullScreenButtonContainer {...args} />;
    return shallow(container);
}

reactTest('ui | containers | buttons | fullscreen-button', function() {
    QUnit.test('renders a FullscreenButton', function(assert) {
        const container = renderContainer();
        assert.equal(container.type(), FullScreenButton);
    });

    QUnit.test('passes in expected props', function(assert) {
        const onFullScreen = sinon.spy();
        const onUnFullScreen = sinon.spy();
        const container = renderContainer({
            canFullScreen: true,
            isFullScreen: true,
            onFullScreen,
            onUnFullScreen,
        });

        container.find(FullScreenButton).props().onFullScreen();

        assert.equal(onFullScreen.callCount, 1, 'onFullScreen prop invoked');
        assert.equal(onUnFullScreen.callCount, 0, 'onUnFullScreen prop not invoked');

        container.find(FullScreenButton).props().onUnFullScreen();

        assert.equal(onFullScreen.callCount, 1, 'onFullScreen not invoked twice');
        assert.equal(onUnFullScreen.callCount, 1, 'onUnFullScreen invoked');

        assert.equal(container.find(FullScreenButton).props().isFullScreen, true, 'isFullScreen prop is passed in');
        assert.equal(container.find(FullScreenButton).props().canFullScreen, true, 'canFullScreen prop is passed in');
    });

    QUnit.test('mapDispatchToProps dispatches expected actions', function(assert) {
        const dispatch = sinon.spy();
        const mappedDispatches = mapDispatchToProps(dispatch);

        mappedDispatches.onFullScreen();
        assert.equal(dispatch.callCount, 2, 'two actions dispatched');
        assert.deepEqual(dispatch.firstCall.args[0], setFullScreen(true), 'fullscreen action dispatched');
        assert.deepEqual(dispatch.secondCall.args[0], trackEvent('player_click_fullscreen'), 'tracking dispatched');

        dispatch.reset();

        mappedDispatches.onUnFullScreen();
        assert.equal(dispatch.callCount, 2, 'two actions dispatched');
        assert.deepEqual(dispatch.firstCall.args[0], setFullScreen(false), 'unfullscreen action dispatched');
        assert.deepEqual(dispatch.secondCall.args[0], trackEvent('player_click_fullscreen'), 'tracking dispatched');
    });

    QUnit.module('mapStateToProps', function() {
        function getState(override) {
            return assign({}, initStore().getState(), override);
        }

        function testProp(propName, expectedValue, expectedState, stateDesc) {
            QUnit.test(`${stateDesc}, ${propName} should be ${expectedValue}`, function(assert) {
                const state = getState(expectedState);
                assert.equal(mapStateToProps(state)[propName], expectedValue);
            });
        }

        testProp('canFullScreen', true, {
            screenMode: {
                canFullScreen: true,
            },
        }, 'when screenMode.canFullScreen is true');

        testProp('canFullScreen', false, {
            screenMode: {
                canFullScreen: false,
            },
        }, 'when screenMode.canFullScreen is false');

        testProp('isFullScreen', true, {
            screenMode: {
                isFullScreen: true,
            },
        }, 'when screenMode.isFullScreen is true');

        testProp('isFullScreen', false, {
            screenMode: {
                isFullScreen: false,
            },
        }, 'when screenMode.isFullScreen is false');

        testProp('hasBranding', true, {
            playerOptions: {
                branding: true,
            },
        }, 'when playerOptions.branding is true');

        testProp('hasBranding', false, {
            playerOptions: {
                branding: false,
            },
        }, 'when playerOptions.branding is false');
    });
});
