import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import {
    PausePlayButtonContainer,
    mapDispatchToProps,
    mapStateToProps,
} from 'ui/containers/buttons/pause-play-button';
import { PausePlayButton } from 'ui/components/buttons/pause-play-button';
import {
    ACTION_VIDEO_API_PLAY,
    ACTION_VIDEO_API_PAUSE,
    ACTION_VIDEO_API_MUTE_PLAYER,
} from 'actions/video-api';
import { ACTION_TRACK_EVENT } from 'actions/analytics-tracker';
import { PLAYER_CLIPS_VIEWING } from 'util/player-type';

const DEFAULT_ARGS = Object.freeze({
    pause() {},
    play() {},
    paused: false,
    showReplay: false,
    unmutePlayer() {},
    trackPlayPause() {},
    isMobileWeb: false,
});

function renderContainer(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const container = <PausePlayButtonContainer {...args} />;
    return shallow(container);
}

reactTest('ui | containers | buttons | pause-play-button', function() {
    QUnit.test('correctly maps state to props', function(assert) {
        const state = {
            playback: {
                paused: true,
                ended: true,
            },
            env: {
                isMobileLocation: true,
                playerType: PLAYER_CLIPS_VIEWING,
            },
        };

        assert.deepEqual(mapStateToProps(state), {
            paused: state.playback.paused,
            showReplay: true,
            isMobileWeb: state.env.isMobileLocation,
        });
    });

    QUnit.test('renders a PausePlayButton', function(assert) {
        const pausePlayButton = renderContainer();
        assert.equal(pausePlayButton.type(), PausePlayButton);
    });

    QUnit.test('passes in paused and showReplay props', function(assert) {
        const pausePlayButton = renderContainer({
            showReplay: true,
            paused: true,
        });

        assert.equal(pausePlayButton.prop('showReplay'), true, 'showReplay passed in');
        assert.equal(pausePlayButton.prop('paused'), true, 'paused passed in');
    });

    QUnit.test('mapDispatchToProps dispatches expected actions', function(assert) {
        const dispatch = sinon.spy();
        const mappedDispatches = mapDispatchToProps(dispatch);

        mappedDispatches.play();
        assert.equal(dispatch.callCount, 1, 'one action dispatched');
        assert.deepEqual(dispatch.firstCall.args[0].type, ACTION_VIDEO_API_PLAY, 'play action dispatched');

        dispatch.reset();

        mappedDispatches.pause();
        assert.equal(dispatch.callCount, 1, 'one action dispatched');
        assert.deepEqual(dispatch.firstCall.args[0].type, ACTION_VIDEO_API_PAUSE, 'pause action dispatched');

        dispatch.reset();

        mappedDispatches.trackPlayPause();
        assert.equal(dispatch.callCount, 1, 'one action dispatched');
        assert.deepEqual(dispatch.firstCall.args[0].type, ACTION_TRACK_EVENT, 'track event action dispatched');
        assert.deepEqual(dispatch.firstCall.args[0].eventName, 'player_click_playpause', 'click playpause event name');

        dispatch.reset();

        mappedDispatches.unmutePlayer();
        assert.equal(dispatch.callCount, 1, 'one action dispatched');
        assert.deepEqual(dispatch.firstCall.args[0].type, ACTION_VIDEO_API_MUTE_PLAYER, 'muted action dispatched');
        assert.deepEqual(dispatch.firstCall.args[0].muted, false, 'muted false value passed in');
        assert.deepEqual(dispatch.firstCall.args[0].automated, false, 'automated false value passed in');
    });

    QUnit.test('onPlay handler calls play and trackPlayPause prop', function(assert) {
        const playSpy = sinon.spy();
        const trackPlayPauseSpy = sinon.spy();
        const pausePlayButton = renderContainer({
            play: playSpy,
            trackPlayPause: trackPlayPauseSpy,
            showReplay: true,
            paused: true,
        });

        playSpy.reset();
        trackPlayPauseSpy.reset();

        const onPlayHandler = pausePlayButton.prop('onPlay');

        onPlayHandler();

        assert.equal(playSpy.callCount, 1, 'play prop called once');
        assert.equal(trackPlayPauseSpy.callCount, 1, 'trackPlayPause prop called once');
    });

    QUnit.test('onPause handler calls pause, trackPlayPause, and if isMobileWeb, unmutePlayer prop', function(assert) {
        const pauseSpy = sinon.spy();
        const trackPlayPauseSpy = sinon.spy();
        const unmuteSpy = sinon.spy();
        const pausePlayButton = renderContainer({
            unmutePlayer: unmuteSpy,
            pause: pauseSpy,
            trackPlayPause: trackPlayPauseSpy,
            showReplay: true,
            paused: true,
            isMobileWeb: false,
        });

        pauseSpy.reset();
        trackPlayPauseSpy.reset();
        unmuteSpy.reset();

        const onPauseHandler = pausePlayButton.prop('onPause');

        onPauseHandler();

        assert.equal(pauseSpy.callCount, 1, 'pause prop called once');
        assert.equal(trackPlayPauseSpy.callCount, 1, 'trackPlayPause prop called once');
        assert.equal(unmuteSpy.callCount, 0, 'unmute prop not called');

        pausePlayButton.setProps({ isMobileWeb: true });

        pauseSpy.reset();
        trackPlayPauseSpy.reset();
        unmuteSpy.reset();

        onPauseHandler();

        assert.equal(pauseSpy.callCount, 1, 'pause prop called once');
        assert.equal(trackPlayPauseSpy.callCount, 1, 'trackPlayPause prop called once');
        assert.equal(unmuteSpy.callCount, 1, 'unmute prop called once');
    });
});
