import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { TheatreModeButtonContainer, mapDispatchToProps } from 'ui/containers/buttons/theatre-mode-button';
import { TheatreModeButton } from 'ui/components/buttons/theatre-mode-button';
import { setTheatreMode } from 'actions/screen-mode';
import { trackEvent } from 'actions/analytics-tracker';

const DEFAULT_ARGS = Object.freeze({
    canTheatreMode: true,
    isTheatreMode: false,
    isFullScreen: false,
    enableTheatreMode() {},
    disableTheatreMode() {},
});

function renderContainer(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const container = <TheatreModeButtonContainer {...args} />;
    return shallow(container);
}

reactTest('ui | containers | buttons | theatre-mode-button', function() {
    QUnit.test('renders a TheatreModeButton', function(assert) {
        const container = renderContainer();
        assert.equal(container.type(), TheatreModeButton);
    });

    QUnit.test('if cannot go into theatre mode, renders nothing', function(assert) {
        const container = renderContainer({
            canTheatreMode: false,
        });

        assert.equal(container.find(TheatreModeButton).length, 0);
    });

    QUnit.test('if is already fullscreen, renders nothing', function(assert) {
        const container = renderContainer({
            isFullScreen: true,
        });

        assert.equal(container.find(TheatreModeButton).length, 0);
    });

    QUnit.test('passes in expected props', function(assert) {
        const enableTheatreMode = sinon.spy();
        const disableTheatreMode = sinon.spy();
        const container = renderContainer({
            canTheatreMode: true,
            isTheatreMode: false,
            enableTheatreMode,
            disableTheatreMode,
        });

        container.find(TheatreModeButton).props().turnOn();

        assert.equal(enableTheatreMode.callCount, 1, 'enableTheatreMode prop invoked');
        assert.equal(disableTheatreMode.callCount, 0, 'disableTheatreMode prop not invoked');

        container.find(TheatreModeButton).props().turnOff();

        assert.equal(enableTheatreMode.callCount, 1, 'enableTheatreMode not invoked twice');
        assert.equal(disableTheatreMode.callCount, 1, 'disableTheatreMode invoked');

        assert.equal(
            container.find(TheatreModeButton).props().theatreModeEnabled,
            false,
            'isTheatreMode prop is passed in'
        );
    });

    QUnit.test('mapDispatchToProps dispatches expected actions', function(assert) {
        const dispatch = sinon.spy();
        const mappedDispatches = mapDispatchToProps(dispatch);

        mappedDispatches.enableTheatreMode();
        assert.equal(dispatch.callCount, 2, 'two actions dispatched');
        assert.deepEqual(dispatch.firstCall.args[0], setTheatreMode(true), 'enable theatre mode action dispatched');
        assert.deepEqual(dispatch.secondCall.args[0], trackEvent('player_click_theater'), 'tracking dispatched');

        dispatch.reset();

        mappedDispatches.disableTheatreMode();
        assert.equal(dispatch.callCount, 2, 'two actions dispatched');
        assert.deepEqual(dispatch.firstCall.args[0], setTheatreMode(false), 'disable theatre mode action dispatched');
        assert.deepEqual(dispatch.secondCall.args[0], trackEvent('player_click_theater'), 'tracking dispatched');
    });
});
