import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { WatchTwitchButtonContainer, mapStateToProps,
         mapDispatchToProps } from 'ui/containers/buttons/watch-twitch-button';
import { WatchTwitchButton } from 'ui/components/buttons/watch-twitch-button';
import { pause } from 'actions/video-api';
import { CONTENT_MODE_LIVE } from 'stream/twitch-live';
import { CONTENT_MODE_VOD } from 'stream/twitch-vod';
import { CONTENT_MODE_NONE } from 'stream/null';
import { CONTENT_MODE_CLIP } from 'stream/clip';
import { mockTranslateFunc } from 'tests/utils/translate-hoc-helpers';
import {
    TRACKING_MEDIUM_EMBED,
    TRACKING_PROPERTY_TWITCH_LOGO,
} from 'ui/player-types/clips/utils/tracking/clips-tracking-constants';

const DEFAULT_PROPS = Object.freeze({
    currentTime: 0,
    streamType: CONTENT_MODE_NONE,
    channelName: 'a channel name',
    videoId: 'a video id',
    windowObj: {},
    showBranding: true,
    metadataUrl: 'someurl.com',
    t: mockTranslateFunc,
});

function renderContainer(overrides = {}) {
    const args = assign({}, DEFAULT_PROPS, overrides);
    const container = <WatchTwitchButtonContainer {...args} />;
    return shallow(container);
}

reactTest('ui | containers | buttons | watch-twitch-button', function() {
    QUnit.test('correctly maps state to props', function(assert) {
        const state = {
            playback: {
                currentTime: 1000,
            },
            stream: {
                contentType: CONTENT_MODE_LIVE,
            },
            streamMetadata: {
                channel: {
                    name: 'a channel name',
                },
                videoId: 'a videoId',
                url: 'a url',
            },
            window: {},
            playerOptions: {
                branding: true,
            },
        };

        assert.deepEqual(mapStateToProps(state), {
            currentTime: state.playback.currentTime,
            streamType: state.stream.contentType,
            channelName: state.streamMetadata.channel.name,
            videoId: state.streamMetadata.videoId,
            metadataUrl: state.streamMetadata.url,
            windowObj: state.window,
            showBranding: state.playerOptions.branding,
        });
    });

    QUnit.test('correctly maps dispatches to props', function(assert) {
        const dispatchSpy = sinon.spy();
        const mappedDispatches = mapDispatchToProps(dispatchSpy);

        mappedDispatches.pausePlayer();

        assert.equal(dispatchSpy.callCount, 1);
        assert.deepEqual(dispatchSpy.firstCall.args[0], pause());
    });

    QUnit.test('shouldComponentUpdate should return false if only the currentTime prop changes', function(assert) {
        const component = renderContainer();
        const updatedProps = assign({}, DEFAULT_PROPS, { currentTime: DEFAULT_PROPS.currentTime + 1 });

        assert.equal(component.instance().shouldComponentUpdate(updatedProps), false);

        updatedProps.channelName = 'another channel name';
        assert.equal(component.instance().shouldComponentUpdate(updatedProps), true);
    });

    QUnit.test('if showBranding prop is false, render null', function(assert) {
        const component = renderContainer({
            showBranding: false,
        });

        assert.equal(component.type(), null);
    });

    QUnit.test('if showBranding is true, render WatchTwitchButton', function(assert) {
        const buttonComponent = renderContainer();
        assert.equal(buttonComponent.type(), WatchTwitchButton);
    });

    // eslint-disable-next-line max-len
    QUnit.test('when current stream type is clip, WatchTwitchButton\'s onClick handler opens twitch.tv, pauses player', function(assert) {
        const windowOpenSpy = sinon.spy();
        const pausePlayerSpy = sinon.spy();
        const buttonComponent = renderContainer({
            windowObj: {
                open: windowOpenSpy,
            },
            pausePlayer: pausePlayerSpy,
            streamType: CONTENT_MODE_CLIP,
        });

        assert.equal(buttonComponent.prop('buttonLabel'), 'View clip page');
        const clickHandler = buttonComponent.prop('onClick');

        assert.equal(windowOpenSpy.callCount, 0, 'has not opened new window');
        assert.equal(pausePlayerSpy.callCount, 0, 'has not called pause yet');

        clickHandler();

        assert.equal(windowOpenSpy.callCount, 1, 'has opened new window');

        const params = `tt_content=${TRACKING_PROPERTY_TWITCH_LOGO}&tt_medium=${TRACKING_MEDIUM_EMBED}`;
        assert.equal(windowOpenSpy.firstCall.args[0],
            `${DEFAULT_PROPS.metadataUrl}?${params}`);
        assert.equal(windowOpenSpy.firstCall.args[1], '_blank', 'opens url in new tab');

        assert.equal(pausePlayerSpy.callCount, 1, 'has paused the video');
    });

    // eslint-disable-next-line max-len
    QUnit.test('when current stream type is none, WatchTwitchButton\'s onClick handler opens twitch.tv, pauses player', function(assert) {
        const windowOpenSpy = sinon.spy();
        const pausePlayerSpy = sinon.spy();
        const buttonComponent = renderContainer({
            windowObj: {
                open: windowOpenSpy,
            },
            pausePlayer: pausePlayerSpy,
        });

        assert.equal(buttonComponent.prop('buttonLabel'), 'Watch on Twitch');
        const clickHandler = buttonComponent.prop('onClick');

        assert.equal(windowOpenSpy.callCount, 0, 'has not opened new window');
        assert.equal(pausePlayerSpy.callCount, 0, 'has not called pause yet');

        clickHandler();

        assert.equal(windowOpenSpy.callCount, 1, 'has opened new window');
        assert.equal(windowOpenSpy.firstCall.args[0], '//www.twitch.tv', 'opens window with correct url');
        assert.equal(windowOpenSpy.firstCall.args[1], '_blank', 'opens url in new tab');

        assert.equal(pausePlayerSpy.callCount, 1, 'has paused the video');
    });

    // eslint-disable-next-line max-len
    QUnit.test('when current stream type is live, WatchTwitchButton\'s onClick handler opens channel page, pauses player', function(assert) {
        const expectedChannelName = 'expectedChannelName';
        const windowOpenSpy = sinon.spy();
        const pausePlayerSpy = sinon.spy();
        const buttonComponent = renderContainer({
            windowObj: {
                open: windowOpenSpy,
            },
            streamType: CONTENT_MODE_LIVE,
            channelName: expectedChannelName,
            pausePlayer: pausePlayerSpy,
        });

        assert.equal(buttonComponent.prop('buttonLabel'), 'Watch on Twitch');
        const clickHandler = buttonComponent.prop('onClick');

        assert.equal(windowOpenSpy.callCount, 0, 'has not opened new window');
        assert.equal(pausePlayerSpy.callCount, 0, 'has not called pause yet');

        clickHandler();

        assert.equal(windowOpenSpy.callCount, 1, 'has opened new window');
        assert.equal(
            windowOpenSpy.firstCall.args[0],
            `//www.twitch.tv/${expectedChannelName}`,
            'opens window with correct url'
        );
        assert.equal(windowOpenSpy.firstCall.args[1], '_blank', 'opens url in new tab');
        assert.equal(pausePlayerSpy.callCount, 1, 'has paused the video');
    });

    // eslint-disable-next-line max-len
    QUnit.test('when current stream type is vod, WatchTwitchButton\'s onClick handler opens vod page with correct timestamp, pauses player', function(assert) {
        const expectedChannelName = 'expectedChannelName';
        const expectedVideoId = '1234567';
        const expectedVideoIdWithV = `v${expectedVideoId}`;
        const expectedURL = `//www.twitch.tv/${expectedChannelName}/v/${expectedVideoId}?t=342h56m07s`;
        const windowOpenSpy = sinon.spy();
        const pausePlayerSpy = sinon.spy();
        const buttonComponent = renderContainer({
            windowObj: {
                open: windowOpenSpy,
            },
            currentTime: 1234567,
            videoId: expectedVideoIdWithV,
            streamType: CONTENT_MODE_VOD,
            pausePlayer: pausePlayerSpy,
            channelName: expectedChannelName,
        });

        assert.equal(buttonComponent.prop('buttonLabel'), 'Watch on Twitch');
        const clickHandler = buttonComponent.prop('onClick');

        assert.equal(windowOpenSpy.callCount, 0, 'has not opened new window');
        assert.equal(pausePlayerSpy.callCount, 0, 'has not called pause yet');

        clickHandler();

        assert.equal(windowOpenSpy.callCount, 1, 'has opened new window');
        assert.equal(windowOpenSpy.firstCall.args[0], expectedURL, 'opens window with correct url');
        assert.equal(windowOpenSpy.firstCall.args[1], '_blank', 'opens url in new tab');
        assert.equal(pausePlayerSpy.callCount, 1, 'has paused the video');
    });
});
