import React from 'react';
import sinon from 'sinon';
import isFunction from 'lodash/isFunction';
import assign from 'lodash/assign';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { BackgroundPanelContainer, mapStateToProps,
         mapDispatchToProps } from 'ui/containers/closed-captions/background-panel';
import { BGandWindowPanel as BackgroundPanelComponent } from 'ui/components/closed-captions/bg-and-window-panel';
import { ACTION_SET_CAPTIONS_PRESET } from 'actions/captions';
import { presetMap } from 'captions/captions-style-map';

const DEFAULT_ARGS = Object.freeze({
    selectColor() {},
    selectOpacity() {},
    currentColor: '',
    currentOpacity: '',
});

const FAKE_STATE = Object.freeze({
    captions: {
        style: {},
        preset: Object.keys(presetMap)[0],
    },
});

function renderBackgroundPanelContainer(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <BackgroundPanelContainer {...args} />;
    return shallow(component);
}

reactTest('ui | containers | closed-captions | background-panel', function() {
    QUnit.test('renders a BackgroundPanelComponent', function(assert) {
        const component = renderBackgroundPanelContainer();
        assert.equal(component.type(), BackgroundPanelComponent);
    });

    function testMapStateToProps(presetValue) {
        QUnit.test('mapStateToProps', function(assert) {
            const state = {
                captions: {
                    style: presetMap[presetValue],
                },
            };

            const props = mapStateToProps(state);

            assert.equal(
                props.currentColor,
                presetMap[presetValue].backgroundColorName,
                `passes ${presetMap[presetValue].backgroundColorName} to props.currentColor`
            );

            assert.equal(
                props.currentOpacity,
                presetMap[presetValue].backgroundOpacity,
                `passes ${presetMap[presetValue].backgroundOpacity} to props.currentOpacity`
            );
        });
    }

    Object.keys(presetMap).forEach(testMapStateToProps);

    function testDispatchMethods(method, modifiedStyle) {
        QUnit.test(`${method} dispatches the correct payload`, function(assert) {
            const dispatch = sinon.spy();
            const methodToTest = mapDispatchToProps(dispatch)[method];
            const data = QUnit.config.current.testId;

            methodToTest(data);
            assert.equal(dispatch.callCount, 1, 'dispatch called once');

            const thunk = dispatch.firstCall.args[0];
            assert.ok(isFunction(thunk), 'a thunk is dispatched');

            dispatch.reset();

            thunk(dispatch, () => FAKE_STATE);
            assert.equal(dispatch.callCount, 1, 'dispatch called once from thunk');

            const action = dispatch.firstCall.args[0];

            assert.equal(
                action.type,
                ACTION_SET_CAPTIONS_PRESET,
                'dispatches a ACTION_SET_CAPTIONS_PRESET action'
            );
            assert.equal(
                action.captions.preset,
                'custom',
                'passes correct preset'
            );
            assert.equal(
                action.captions.style[modifiedStyle],
                data,
                `sets ${modifiedStyle} correctly`
            );
        });
    }

    testDispatchMethods('selectColor', 'backgroundColorName');
    testDispatchMethods('selectOpacity', 'backgroundOpacity');
});
