import React from 'react';
import { reactTest } from 'tests/utils/react-test';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { shallow } from 'enzyme';
import {
    ClosedCaptionsPanelContainer, mapStateToProps, mapDispatchToProps,
} from 'ui/containers/closed-captions/closed-captions-panel';
import {
    ClosedCaptionsPanel as ClosedCaptionsPanelComponent,
} from 'ui/components/closed-captions/closed-captions-panel';
import { showCCModal } from 'actions/captions';

const DEFAULT_ARGS = Object.freeze({
    onClose() {},
    showPanel: false,
});

function renderCCpanel(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <ClosedCaptionsPanelContainer {...args} />;
    return shallow(component);
}

reactTest('ui | containers | closed-captions | closed-captions-panel', function() {
    QUnit.test('renders null if showPanel is false', function(assert) {
        const component = renderCCpanel();
        assert.equal(component.type(), null, 'renders null when showPanel is false');
    });

    QUnit.test('renders a ClosedCaptionsPanelComponent', function(assert) {
        const component = renderCCpanel({ showPanel: true });
        assert.equal(component.type(), ClosedCaptionsPanelComponent);
    });

    QUnit.test('passes onClose to ClosedCaptionsPanelComponent', function(assert) {
        const component = renderCCpanel({ showPanel: true });
        const { onClose } = component.props();
        assert.deepEqual(onClose, DEFAULT_ARGS.onClose, 'passes onClose to component');
    });

    QUnit.test('handleClickOutside invokes onClose if showPanel is true', function(assert) {
        const onClose = sinon.spy();
        const component = renderCCpanel({
            onClose,
            showPanel: true,
        });
        component.instance().handleClickOutside();

        assert.equal(onClose.callCount, 1, 'called once');
    });

    QUnit.test('handleClickOutside does not call onClose if showPanel is false', function(assert) {
        const onClose = sinon.spy();
        const component = renderCCpanel({
            onClose,
            showPanel: false,
        });
        component.instance().handleClickOutside();

        assert.equal(onClose.callCount, 0, 'not called');
    });

    QUnit.test('mapStateToProps - passes showCCModal to showPanel', function(assert) {
        const captions = {
            showCCModal: true,
        };
        const { showPanel } = mapStateToProps({ captions });
        assert.equal(showPanel, true, 'showCCModal sets showPanel correctly');
    });

    QUnit.test('mapDispatchToProps - onClose dispatches expected action', function(assert) {
        const dispatch = sinon.spy();
        const { onClose } = mapDispatchToProps(dispatch);
        onClose();
        assert.equal(dispatch.callCount, 1, 'called once');
        assert.deepEqual(
            dispatch.firstCall.args[0],
            showCCModal(false),
            'passes expected action'
        );
    });
});
