import React from 'react';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import {
    EffectsPanelContainer, mapStateToProps, mapDispatchToProps,
} from 'ui/containers/closed-captions/effects-panel';
import { EffectsPanel as EffectsPanelComponent } from 'ui/components/closed-captions/effects-panel';
import * as CaptionsActions from 'actions/captions';

const DEFAULT_ARGS = Object.freeze({
    boldText: false,
    italicText: false,
    underlineText: false,
    currentTextEffect: 'none',
    selectTextEffect() {},
    selectTextStyle() {},
});

reactTest('ui | containers | closed-captions | effects-panel', function(hooks) {
    hooks.beforeEach(function() {
        this._mockSetCaptionsPreset = sinon.spy(CaptionsActions, 'setCaptionsPreset');
    });

    hooks.afterEach(function() {
        this._mockSetCaptionsPreset.restore();
    });

    QUnit.test('should render an EffectsPanelComponent', function(assert) {
        const component = shallow(<EffectsPanelContainer {...DEFAULT_ARGS} />);
        assert.equal(component.type(), EffectsPanelComponent);
    });

    QUnit.test('should render with correct props', function(assert) {
        const component = shallow(<EffectsPanelContainer {...DEFAULT_ARGS} />);
        const props = component.props();
        assert.deepEqual(props, DEFAULT_ARGS, 'has correct props');
    });

    QUnit.test('mapDispatchToProps - selectTextEffect dispatches correct action', function(assert) {
        const dispatch = sinon.spy();
        const { selectTextEffect } = mapDispatchToProps(dispatch);
        const effect = QUnit.config.current.testId;

        selectTextEffect(effect);

        assert.equal(dispatch.callCount, 1, 'called once');

        assert.equal(this._mockSetCaptionsPreset.firstCall.args[0], 'custom', 'first arg is the custom preset');
        assert.deepEqual(
            this._mockSetCaptionsPreset.firstCall.args[1],
            {
                edge: effect,
            },
            'passes correct payload'
        );
    });

    QUnit.test('mapDispatchToProps - selectTextStyle dispatches correct action', function(assert) {
        const dispatch = sinon.spy();
        const { selectTextStyle } = mapDispatchToProps(dispatch);
        const style = QUnit.config.current.testId;
        const propName = QUnit.config.current.testId.split().reverse().join('');

        selectTextStyle(propName, style);

        assert.equal(dispatch.callCount, 1, 'called once');

        assert.equal(this._mockSetCaptionsPreset.firstCall.args[0], 'custom', 'first arg is the custom preset');
        assert.deepEqual(
            this._mockSetCaptionsPreset.firstCall.args[1],
            {
                [propName]: style,
            },
            'passes correct payload'
        );
    });

    QUnit.test('mapStateToProps - gets the correct values from store', function(assert) {
        const fakeState = {
            captions: {
                style: {
                    fontBold: 'bold',
                    fontItalic: 'italic',
                    fontUnderline: 'none',
                    edge: 'my random edge',
                },
            },
        };
        const props = mapStateToProps(fakeState);

        assert.deepEqual(props, {
            boldText: true,
            italicText: true,
            underlineText: false,
            currentTextEffect: fakeState.captions.style.edge,
        }, 'creates correct props');
    });
});
