import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { PresetsPanelContainer, mapDispatchToProps,
         mapStateToProps } from 'ui/containers/closed-captions/presets-panel';
import { PresetsPanel as PresetsPanelComponent, CC_PRESET_VALUES } from 'ui/components/closed-captions/presets-panel';
import * as StyleMap from 'captions/captions-style-map';
import { ACTION_SET_CAPTIONS_PRESET } from 'actions/captions';

const DEFAULT_ARGS = Object.freeze({
    onSelect() {},
    selectedValue: CC_PRESET_VALUES[0],
});

const FAKE_STATE = Object.freeze({
    captions: {
        style: {},
        preset: CC_PRESET_VALUES[0],
    },
});

function renderPresetsPanelContainer(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <PresetsPanelContainer {...args} />;
    return shallow(component);
}

reactTest('ui | containers | closed-captions | presets-panel', function() {
    QUnit.test('renders a PresetsPanelComponent', function(assert) {
        const component = renderPresetsPanelContainer();
        assert.equal(component.type(), PresetsPanelComponent);
    });

    QUnit.test('passes onSelect to component', function(assert) {
        const onSelect = sinon.spy();
        const component = renderPresetsPanelContainer({ onSelect });
        assert.strictEqual(component.props().onSelect, onSelect);
    });

    QUnit.test('passes selectedValue to component', function(assert) {
        const selectedValue = CC_PRESET_VALUES[1];
        const component = renderPresetsPanelContainer({ selectedValue });
        assert.equal(component.props().selectedValue, selectedValue);
    });

    function testMapStateToProps(preset) {
        QUnit.test('mapStateToProps', function(assert) {
            const state = {
                captions: {
                    preset,
                },
            };

            const props = mapStateToProps(state);

            assert.equal(props.selectedValue, preset, `passes ${preset} to props`);
        });
    }

    CC_PRESET_VALUES.forEach(testMapStateToProps);

    function testDispatch(presetValue) {
        QUnit.test(`dispatches setCaptionsPreset with ${presetValue} when onSelect is invoked`, function(assert) {
            const dispatch = sinon.spy();
            const { onSelect } = mapDispatchToProps(dispatch);

            onSelect(presetValue);

            assert.equal(dispatch.callCount, 1, 'dispatch called once');

            const thunk = dispatch.firstCall.args[0];
            dispatch.reset();

            thunk(dispatch, () => FAKE_STATE);

            assert.equal(dispatch.callCount, 1, 'dispatch called once from thunk');
            const action = dispatch.firstCall.args[0];

            assert.equal(
                action.type,
                ACTION_SET_CAPTIONS_PRESET,
                'dispatches a ACTION_SET_CAPTIONS_PRESET action'
            );
            assert.equal(
                action.captions.preset,
                presetValue,
                'passes correct preset'
            );
            assert.deepEqual(
                action.captions.style,
                StyleMap.presetMap[presetValue],
                'passes correct styles'
            );
        });
    }

    CC_PRESET_VALUES.forEach(testDispatch);
});
