import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { TextPanelContainer, mapStateToProps, mapDispatchToProps } from 'ui/containers/closed-captions/text-panel';
import { presetMap } from 'captions/captions-style-map';
import { TextPanel as TextPanelComponent } from 'ui/components/closed-captions/text-panel';
import { ACTION_SET_CAPTIONS_PRESET } from 'actions/captions';

const DEFAULT_STYLE = Object.freeze(assign({}, presetMap['white-on-black']));

const FAKE_STATE = Object.freeze({
    captions: {
        style: DEFAULT_STYLE,
    },
});

const DEFAULT_ARGS = Object.freeze({
    currentAlignment: '',
    currentColor: '',
    currentFont: '',
    currentFontSize: 0,
    currentPosition: '',
    currentOpacity: '',
    selectFontChange() {},
});

function renderTextPanelContainer(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <TextPanelContainer {...args} />;
    return shallow(component);
}

reactTest('ui | containers | closed-captions | text-panel', function() {
    QUnit.test('renders TextPanelComponent', function(assert) {
        const component = renderTextPanelContainer();
        assert.equal(component.type(), TextPanelComponent);
    });

    QUnit.test('correct props are passed to TextPanelComponent', function(assert) {
        const mockProps = {
            currentAlignment: 'left',
            currentColor: 'red',
            currentFont: 'mono-serif',
            currentFontSize: 20,
            currentPosition: 'top',
            currentOpacity: 'solid',
            selectFontChange() {},
        };
        const component = renderTextPanelContainer(mockProps);

        assert.deepEqual(component.props(), mockProps, 'passes correct props to component');
    });

    function testStateToProps({ styleProp, value }, mappedStateProp) {
        QUnit.test(`converts ${styleProp} to ${mappedStateProp} correctly`, function(assert) {
            const style = assign({}, DEFAULT_STYLE, {
                [styleProp]: value,
            });
            const state = {
                captions: { style },
            };

            const props = mapStateToProps(state);
            assert.equal(props[mappedStateProp], value, `${mappedStateProp} is set to ${value}`);
        });
    }

    testStateToProps({
        styleProp: 'alignment',
        value: 'randomValue',
    }, 'currentAlignment');

    testStateToProps({
        styleProp: 'fontColorName',
        value: 'randomValue',
    }, 'currentColor');

    testStateToProps({
        styleProp: 'font',
        value: 'randomValue',
    }, 'currentFont');

    testStateToProps({
        styleProp: 'fontSize',
        value: 20,
    }, 'currentFontSize');

    testStateToProps({
        styleProp: 'verticalPosition',
        value: 'randomValue',
    }, 'currentPosition');

    testStateToProps({
        styleProp: 'fontOpacity',
        value: 'randomValue',
    }, 'currentOpacity');

    QUnit.test('mapDispatchToProps - selectFontChange dispatches correct action', function(assert) {
        const dispatch = sinon.spy();
        const { selectFontChange } = mapDispatchToProps(dispatch);
        const fakeProp = 'font';
        const fakeValue = QUnit.config.current.testId;

        selectFontChange(fakeProp, fakeValue);
        assert.equal(dispatch.callCount, 1, 'called once');

        const thunk = dispatch.firstCall.args[0];

        dispatch.reset();

        thunk(dispatch, () => FAKE_STATE);

        assert.equal(dispatch.callCount, 1, 'called once from thunk');
        assert.deepEqual(dispatch.firstCall.args[0], {
            type: ACTION_SET_CAPTIONS_PRESET,
            captions: {
                preset: 'custom',
                style: assign({}, DEFAULT_STYLE, {
                    [fakeProp]: fakeValue,
                }),
            },
        }, 'correct styles changed');
    });
});
