import React from 'react';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { CollectionSideBarContainer,
         mapStateToProps, mapDispatchToProps } from 'ui/containers/collections/collection-sidebar';
import { AnimatedCollectionSidebar } from 'ui/components/collections/collection-sidebar';
import { TEST_NORMALIZED_COLLECTION } from 'tests/fixtures/collection';
import { ACTION_CLOSE_COLLECTION_SIDEBAR } from 'actions/collection';
import assign from 'lodash/assign';
import sinon from 'sinon';
import TransitionGroup from 'react-transition-group/TransitionGroup';
import * as PlaybackActions from 'actions/playback';
import { SIDEBAR_VIEW } from 'state/collection';
import { STITCHED_ADVERTISEMENT_SCREEN, ADVERTISEMENT_SCREEN, CONTENT_SCREEN } from 'actions/screen';
import { CONTENT_MODE_VOD } from 'stream/twitch-vod';
import { CONTENT_MODE_LIVE } from 'stream/twitch-live';

const DEFAULT_PROPS = Object.freeze({
    collection: TEST_NORMALIZED_COLLECTION,
    showSidebar: true,
    currentVideoId: '',
    onCloseSidebar: () => {},
    onSelectCollectionItem: () => {},
    windowObj: {},
    shouldRender: true,
});

function renderSidebarContainer(overrides = {}) {
    const args = assign({}, DEFAULT_PROPS, overrides);
    const component = <CollectionSideBarContainer {...args} />;
    return shallow(component);
}

// a fake selectCollectionVideo action
const FAKE_SELECT_COLLECTION_VIDEO_ACTION = 'a fake select collection video action';

reactTest('ui | containers | collections | collection-sidebar', function(hooks) {
    hooks.beforeEach(function() {
        sinon.stub(PlaybackActions, 'selectCollectionVideo', (vodID, collectionID) => ({
            vodID,
            collectionID,
            type: FAKE_SELECT_COLLECTION_VIDEO_ACTION,
        }));
    });

    hooks.afterEach(function() {
        PlaybackActions.selectCollectionVideo.restore();
    });

    QUnit.test('correctly maps state to props - if contentType is vod', function(assert) {
        const state = {
            collection: TEST_NORMALIZED_COLLECTION,
            stream: {
                videoId: 'a stream video id',
                contentType: CONTENT_MODE_VOD,
            },
            window: {},
            ui: {
                isMini: false,
            },
            screen: [CONTENT_SCREEN],
        };

        assert.deepEqual(mapStateToProps(state), {
            collection: state.collection,
            showSidebar: state.collection.currentView === SIDEBAR_VIEW,
            currentVideoId: state.stream.videoId,
            windowObj: state.window,
            shouldRender: true,
        });

        state.ui = { isMini: true };
        assert.equal(mapStateToProps(state).shouldRender, false);
        state.ui = { isMini: false };

        state.screen = [STITCHED_ADVERTISEMENT_SCREEN];
        assert.equal(mapStateToProps(state).shouldRender, false);
        state.screen = [CONTENT_SCREEN];

        state.screen = [ADVERTISEMENT_SCREEN];
        assert.equal(mapStateToProps(state).shouldRender, false);
        state.screen = [CONTENT_SCREEN];

        state.collection.id = '';
        assert.equal(mapStateToProps(state).shouldRender, false);
    });

    // eslint-disable-next-line max-len
    QUnit.test('sets shouldRender to false, and currentVideoId to empty string if not a vod', function(assert) {
        const state = {
            collection: TEST_NORMALIZED_COLLECTION,
            stream: {
                videoId: 'a stream video id',
                contentType: CONTENT_MODE_LIVE,
            },
            window: {},
            ui: {
                isMini: false,
            },
            screen: [CONTENT_SCREEN],
        };

        assert.equal(mapStateToProps(state).currentVideoId, '');
        assert.equal(mapStateToProps(state).shouldRender, false);
    });

    QUnit.test('correctly maps dispatches to props', function(assert) {
        const dispatchSpy = sinon.spy();
        const dispatchProps = mapDispatchToProps(dispatchSpy);
        const collectionItemID = 'a collection item id';
        const collectionID = 'a collection id';

        dispatchProps.onCloseSidebar();
        assert.equal(dispatchSpy.firstCall.args[0].type, ACTION_CLOSE_COLLECTION_SIDEBAR);

        dispatchSpy.reset();

        dispatchProps.onSelectCollectionItem(collectionItemID, collectionID);
        assert.equal(dispatchSpy.firstCall.args[0].type, FAKE_SELECT_COLLECTION_VIDEO_ACTION);
        assert.equal(dispatchSpy.firstCall.args[0].vodID, `v${collectionItemID}`);
        assert.equal(dispatchSpy.firstCall.args[0].collectionID, collectionID);
    });
    // eslint-disable-next-line max-len
    QUnit.test('render animated sidebar wrapped in transition group when showSidebar is true and collections are shown', function(assert) {
        const reactTransitionGroupComponent = renderSidebarContainer();
        const animatedSidebarComponent = reactTransitionGroupComponent.childAt(0);

        assert.equal(reactTransitionGroupComponent.type(), TransitionGroup);
        assert.equal(reactTransitionGroupComponent.prop('component'), 'div');
        assert.equal(animatedSidebarComponent.type(), AnimatedCollectionSidebar);
        assert.equal(animatedSidebarComponent.prop('collection'), DEFAULT_PROPS.collection);
        assert.equal(animatedSidebarComponent.prop('onClose'), DEFAULT_PROPS.onCloseSidebar);
    });

    QUnit.test('sets currentItemIndex prop on the animated sidebar correctly', function(assert) {
        const expectedIndex = 1;
        const component = renderSidebarContainer({
            currentVideoId: `v${TEST_NORMALIZED_COLLECTION.items[expectedIndex].itemId}`,
        });

        const animatedSidebarComponent = component.childAt(0);
        assert.equal(animatedSidebarComponent.prop('currentItemIndex'), expectedIndex);
    });

    QUnit.test('sets key on the animated sidebar correctly', function(assert) {
        const expectedIndex = 1;

        const component = renderSidebarContainer({
            currentVideoId: `v${TEST_NORMALIZED_COLLECTION.items[expectedIndex].itemId}`,
        });

        const animatedSidebarComponent = component.childAt(0);
        assert.equal(animatedSidebarComponent.key(), `${DEFAULT_PROPS.collection.id}_${expectedIndex}`);
    });

    // eslint-disable-next-line max-len
    QUnit.test('the onMouseLeave prop on animated sidebar sets a timeout of 400 ms that calls the onCloseSidebar prop', function(assert) {
        const setTimeoutSpy = sinon.spy();
        const onCloseSidebarSpy = sinon.spy();
        const component = renderSidebarContainer({
            windowObj: {
                setTimeout: setTimeoutSpy,
            },
            onCloseSidebar: onCloseSidebarSpy,
        });

        const animatedSidebarComponent = component.childAt(0);
        const mouseLeaveHandler = animatedSidebarComponent.prop('onMouseLeave');
        assert.equal(setTimeoutSpy.callCount, 0);
        assert.equal(onCloseSidebarSpy.callCount, 0);

        mouseLeaveHandler();

        assert.equal(setTimeoutSpy.callCount, 1);
        assert.equal(setTimeoutSpy.firstCall.args[1], 400, 'timeout period is 400 ms');

        setTimeoutSpy.firstCall.args[0]();

        assert.equal(onCloseSidebarSpy.callCount, 1);
    });

    QUnit.test('the onMouseEnter prop on animated sidebar clears sidebar closing timout', function(assert) {
        const timeoutID = 'a timeout id';
        const setTimeoutStub = sinon.stub().returns(timeoutID);
        const clearTimeoutSpy = sinon.spy();

        const component = renderSidebarContainer({
            windowObj: {
                setTimeout: setTimeoutStub,
                clearTimeout: clearTimeoutSpy,
            },
        });

        assert.equal(clearTimeoutSpy.callCount, 0);

        const animatedSidebarComponent = component.childAt(0);
        const mouseEnterHandler = animatedSidebarComponent.prop('onMouseEnter');
        const mouseLeaveHandler = animatedSidebarComponent.prop('onMouseLeave');

        mouseLeaveHandler();
        mouseEnterHandler();

        assert.equal(clearTimeoutSpy.callCount, 1);
        assert.equal(clearTimeoutSpy.firstCall.args[0], timeoutID);
    });

    // eslint-disable-next-line max-len
    QUnit.test('onSelectItem prop on animated sidebar calls onSelectCollectionItem with correct collection id', function(assert) {
        const onSelectCollectionItemSpy = sinon.spy();
        const testItemID = 'a test item id';
        const component = renderSidebarContainer({
            onSelectCollectionItem: onSelectCollectionItemSpy,
        });

        const animatedSidebarComponent = component.childAt(0);
        const onSelectItemProp = animatedSidebarComponent.prop('onSelectItem');

        onSelectItemProp(testItemID);

        assert.equal(onSelectCollectionItemSpy.callCount, 1);
        assert.equal(onSelectCollectionItemSpy.firstCall.args[0], testItemID);
        assert.equal(onSelectCollectionItemSpy.firstCall.args[1], DEFAULT_PROPS.collection.id);
    });

    // eslint-disable-next-line max-len
    QUnit.test('render null wrapped in transition group when showSidebar is false, but collections are shown', function(assert) {
        const reactTransitionGroupComponent = renderSidebarContainer({
            showSidebar: false,
            isShowingCollection: true,
        });

        assert.equal(reactTransitionGroupComponent.type(), TransitionGroup);
        assert.equal(reactTransitionGroupComponent.prop('component'), 'div');
        assert.equal(reactTransitionGroupComponent.childAt(0).type(), null);
    });

    QUnit.test('render null when shouldRender is false', function(assert) {
        const reactTransitionGroupComponent = renderSidebarContainer({
            shouldRender: false,
        });

        assert.equal(reactTransitionGroupComponent.type(), null);
    });

    QUnit.test('when the container component unmounts, clears the sidebar closing timeout', function(assert) {
        const timeoutID = 'a timeout id';
        const setTimeoutStub = sinon.stub().returns(timeoutID);
        const clearTimeoutSpy = sinon.spy();

        const component = renderSidebarContainer({
            windowObj: {
                setTimeout: setTimeoutStub,
                clearTimeout: clearTimeoutSpy,
            },
        });

        assert.equal(clearTimeoutSpy.callCount, 0);

        const animatedSidebarComponent = component.childAt(0);
        const mouseLeaveHandler = animatedSidebarComponent.prop('onMouseLeave');

        mouseLeaveHandler();
        component.instance().componentWillUnmount();

        assert.equal(clearTimeoutSpy.callCount, 1);
        assert.equal(clearTimeoutSpy.firstCall.args[0], timeoutID);
    });

    QUnit.test('handleClickOutside calls the onCloseSidebar prop when showSidebar is true', function(assert) {
        const onCloseSidebarSpy = sinon.spy();
        const component = renderSidebarContainer({
            onCloseSidebar: onCloseSidebarSpy,
        });

        assert.equal(onCloseSidebarSpy.callCount, 0);

        component.instance().handleClickOutside();

        assert.equal(onCloseSidebarSpy.callCount, 1);
    });
});
