import React from 'react';
import assign from 'lodash/assign';
import merge from 'lodash/merge';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { CollectionTopBarContainer, mapStateToProps,
         mapDispatchToProps } from 'ui/containers/collections/collection-topbar';
import { CollectionTopBar as CollectionTopBarComponent } from 'ui/components/collections/collection-topbar';
import { COLLAPSED_VIEW, SIDEBAR_VIEW } from 'state/collection';
import { ACTION_OPEN_COLLECTION_SIDEBAR } from 'actions/collection';
import { CONTENT_SCREEN } from 'actions/screen';
import { CONTENT_MODE_VOD } from 'stream/twitch-vod';
import { CONTENT_MODE_LIVE } from 'stream/twitch-live';

const DEFAULT_ARGS = Object.freeze({
    collectionItems: [],
    shouldShowTopBar: true,
    showDivider: false,
    stream: {
        videoId: 'v123435',
        contentType: CONTENT_MODE_VOD,
    },
    title: '',
    win: {
        setTimeout() {},
        clearTimeout() {},
    },
});

function renderCollectionTopBarContainer(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <CollectionTopBarContainer {...args} />;
    return shallow(component);
}

reactTest('ui | containers | collections | collection-topbar', function() {
    QUnit.test('if shouldShowTopBar is false, render null', function(assert) {
        const component = renderCollectionTopBarContainer({
            shouldShowTopBar: false,
        });

        assert.equal(component.type(), null);
    });

    QUnit.test('if not vod stream, render null', function(assert) {
        const component = renderCollectionTopBarContainer({
            stream: {
                contentType: CONTENT_MODE_LIVE,
            },
        });

        assert.equal(component.type(), null);
    });

    QUnit.test('if shouldShowTopBar is true, render CollectionTopBarComponent', function(assert) {
        const component = renderCollectionTopBarContainer();
        assert.equal(component.type(), CollectionTopBarComponent);
    });

    QUnit.test('The correct props are passed to CollectionTopBarComponent', function(assert) {
        const component = renderCollectionTopBarContainer();
        const expectedProps = {
            onMouseEnter: component.instance().onMouseEnter,
            onMouseLeave: component.instance().onMouseLeave,
            collectionItems: DEFAULT_ARGS.collectionItems,
            showDivider: DEFAULT_ARGS.showDivider,
            showSidebar: component.instance().showSidebar,
            streamId: String(DEFAULT_ARGS.stream.videoId).substr(1),
            title: DEFAULT_ARGS.title,
        };

        assert.deepEqual(component.props(), expectedProps);
    });

    QUnit.test('invoking this.showSidebar invokes this.props.showSidebar', function(assert) {
        const showSidebar = sinon.spy();
        const component = renderCollectionTopBarContainer({ showSidebar });

        component.instance().showSidebar();

        assert.equal(showSidebar.callCount, 1);
    });

    QUnit.test('onMouseEnter creates a setTimeout with the appropriate arguments', function(assert) {
        const timeoutId = parseInt((Math.random() * 100), 10);
        const win = {
            setTimeout: sinon.stub().returns(timeoutId),
        };
        const component = renderCollectionTopBarContainer({ win });
        component.instance().onMouseEnter();

        assert.equal(win.setTimeout.callCount, 1);
        assert.equal(component.instance()._hoverTimeout, timeoutId, 'Creates a setTimeout ID');
        assert.deepEqual(
            win.setTimeout.firstCall.args[0],
            component.instance().showSidebar,
            'passes correct callback'
        );
        assert.equal(win.setTimeout.firstCall.args[1], 250, '250ms timeout');
    });

    QUnit.test('onMouseLeave clears the timeout created by onMouseEnter', function(assert) {
        const timeoutId = parseInt((Math.random() * 100), 10);
        const win = {
            setTimeout: sinon.stub().returns(timeoutId),
            clearTimeout: sinon.spy(),
        };
        const component = renderCollectionTopBarContainer({ win });
        component.instance().onMouseEnter();

        assert.equal(
            component.instance()._hoverTimeout,
            timeoutId,
            'Creates a setTimeout ID'
        );

        component.instance().onMouseLeave();

        assert.equal(win.clearTimeout.callCount, 1);
        assert.equal(win.clearTimeout.firstCall.args[0], timeoutId);
        assert.equal(component.instance()._hoverTimeout, 0, 'reset to 0');
    });

    QUnit.module('mapStateToProps', function(hooks) {
        hooks.beforeEach(function() {
            this.fakeState = {
                collection: {
                    title: 'fake title',
                    currentView: COLLAPSED_VIEW,
                    items: [],
                },
                playerOptions: {
                    showInfo: true,
                },
                stream: {
                    videoId: 'v12354',
                    contentType: CONTENT_MODE_VOD,
                },
                window: {},
                ui: {
                    isMini: false,
                },
                screenMode: {
                    isTheatreMode: false,
                },
                screen: [CONTENT_SCREEN],
            };
        });

        QUnit.test('mapStateToProps converts state to props properly', function(assert) {
            const resultProps = mapStateToProps(this.fakeState);
            const shouldShowTopBar = this.fakeState.collection.currentView === COLLAPSED_VIEW &&
                                     !this.fakeState.ui.isMini &&
                                     this.fakeState.screen[0] === CONTENT_SCREEN;

            const expectedProps = {
                collectionItems: this.fakeState.collection.items,
                shouldShowTopBar,
                showDivider: this.fakeState.playerOptions.showInfo || this.fakeState.screenMode.isTheatreMode,
                stream: this.fakeState.stream,
                title: this.fakeState.collection.title,
                win: this.fakeState.window,
            };

            assert.deepEqual(resultProps, expectedProps);
        });

        QUnit.test('showDivider is true if theatreMode, and showInfo is false', function(assert) {
            const resultProps = mapStateToProps(merge({}, this.fakeState, {
                playerOptions: {
                    showInfo: false,
                },
                screenMode: {
                    isTheatreMode: true,
                },
            }));

            assert.equal(resultProps.showDivider, true);
        });

        QUnit.test('showDivider is false if not theatreMode, and showInfo is false', function(assert) {
            const resultProps = mapStateToProps(merge({}, this.fakeState, {
                playerOptions: {
                    showInfo: false,
                },
                screenMode: {
                    isTheatreMode: false,
                },
            }));

            assert.equal(resultProps.showDivider, false);
        });

        QUnit.test('showDivider is true if not theatreMode, and showInfo is true', function(assert) {
            const resultProps = mapStateToProps(merge({}, this.fakeState, {
                playerOptions: {
                    showInfo: true,
                },
                screenMode: {
                    isTheatreMode: false,
                },
            }));

            assert.equal(resultProps.showDivider, true);
        });

        // eslint-disable-next-line max-len
        QUnit.test('shouldShowTopBar is false if collection.currentView is not collapsed and !ui.isMini', function(assert) {
            const resultProps = mapStateToProps(merge({}, this.fakeState, {
                collection: {
                    currentView: SIDEBAR_VIEW,
                },
                ui: {
                    isMini: false,
                },
            }));

            assert.equal(resultProps.shouldShowTopBar, false);
        });

        QUnit.test('shouldShowTopBar is false if screen[0] is not content screen', function(assert) {
            const resultProps = mapStateToProps(merge({}, this.fakeState, {
                screen: ['random screen'],
            }));
            assert.equal(resultProps.shouldShowTopBar, false);
        });

        QUnit.test('shouldShowTopBar is true if screen[0] is content screen', function(assert) {
            const resultProps = mapStateToProps(merge({}, this.fakeState, {
                screen: [CONTENT_SCREEN],
            }));
            assert.equal(resultProps.shouldShowTopBar, true);
        });

        // eslint-disable-next-line max-len
        QUnit.test('shouldShowTopBar is true if collection.currentView is collapsed and !ui.isMini', function(assert) {
            const resultProps = mapStateToProps(merge({}, this.fakeState, {
                collection: {
                    currentView: COLLAPSED_VIEW,
                },
                ui: {
                    isMini: false,
                },
            }));

            assert.equal(resultProps.shouldShowTopBar, true);
        });

        // eslint-disable-next-line max-len
        QUnit.test('shouldShowTopBar is false if collection.currentView is collapsed and ui.isMini', function(assert) {
            const resultProps = mapStateToProps(merge({}, this.fakeState, {
                collection: {
                    currentView: COLLAPSED_VIEW,
                },
                ui: {
                    isMini: true,
                },
            }));

            assert.equal(resultProps.shouldShowTopBar, false);
        });
    });

    QUnit.module('mapDispatchToProps', function() {
        QUnit.test('maps showSidebar to openCollectionSidebar action', function(assert) {
            const dispatch = sinon.spy();
            const props = mapDispatchToProps(dispatch);

            assert.ok(props.hasOwnProperty('showSidebar'));

            props.showSidebar();

            assert.equal(dispatch.callCount, 1);
            assert.deepEqual(dispatch.firstCall.args[0], {
                type: ACTION_OPEN_COLLECTION_SIDEBAR,
            });
        });
    });
});
