import React from 'react';
import merge from 'lodash/merge';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { FeaturedCollectionContainer } from 'ui/containers/collections/featured-collection';
import { FeaturedCollectionComponent } from 'ui/components/collections/featured-collection';

const DEFAULT_ARGS = {
    featuredCollection: {
        id: '',
        title: '',
        itemsCount: 0,
        owner: {
            displayName: '',
            id: '',
            name: '',
        },
        thumbnails: {
            large: '',
            medium: '',
            small: '',
            template: '',
        },
    },
    showFeaturedCollection: false,
};

function renderFeaturedCollectionContainer(overrides = {}) {
    const args = merge({}, DEFAULT_ARGS, overrides);
    const component = <FeaturedCollectionContainer {...args} />;
    return shallow(component);
}

reactTest('ui | containers | collections | featured-collection', function() {
    QUnit.test('renders null if showFeaturedCollection is false', function(assert) {
        const component = renderFeaturedCollectionContainer();
        assert.equal(component.type(), null);
    });

    QUnit.test('renders a FeaturedCollectionComponent if showFeaturedCollection is true', function(assert) {
        const component = renderFeaturedCollectionContainer({ showFeaturedCollection: true });
        assert.equal(component.type(), FeaturedCollectionComponent);
    });

    QUnit.test('passes the correct props to FeaturedCollectionComponent', function(assert) {
        const thumbnails = {
            large: 'large-url',
            medium: 'medium-url',
            small: 'small-url',
            template: 'template-url',
        };
        const component = renderFeaturedCollectionContainer({
            featuredCollection: { thumbnails },
            showFeaturedCollection: true,
        });
        const instance = component.instance();

        assert.deepEqual(component.props(), {
            thumbnailURL: thumbnails.medium,
            featuredCollection: merge({}, DEFAULT_ARGS.featuredCollection, { thumbnails }),
            onSelect: instance.handleSelect,
        });
    });

    QUnit.test('handleSelect fires requestCollection and logs a tracking event', function(assert) {
        const requestCollection = sinon.spy();
        const trackEvent = sinon.spy();
        const component = renderFeaturedCollectionContainer({
            showFeaturedCollection: true,
            trackEvent,
            requestCollection,
        });

        component.instance().handleSelect({ id: QUnit.config.current.testId });

        assert.equal(requestCollection.callCount, 1);
        assert.equal(requestCollection.firstCall.args[0], QUnit.config.current.testId);

        assert.equal(trackEvent.callCount, 1);
        assert.equal(trackEvent.firstCall.args[0], 'player_rec_select');
        assert.deepEqual(trackEvent.firstCall.args[1], {
            // eslint-disable-next-line camelcase
            recommended_vod_view: 'collection',
            // eslint-disable-next-line camelcase
            recommended_collection_id: QUnit.config.current.testId,
        });
    });

    QUnit.test('When showFeaturedCollection changes to true, fire tracking event, preload images', function(assert) {
        const trackEvent = sinon.spy();
        const component = renderFeaturedCollectionContainer({
            trackEvent,
        });
        const instance = component.instance();
        sinon.spy(instance, '_preloadThumbnail');

        component.setProps({
            showFeaturedCollection: true,
            featuredCollection: {
                id: QUnit.config.current.testId,
                thumbnails: {
                    large: 'large-url',
                    medium: 'medium-url',
                    small: 'small-url',
                    template: 'template-url',
                },
            },
        });

        assert.equal(trackEvent.callCount, 1);
        assert.equal(trackEvent.firstCall.args[0], 'player_rec_show_collection');
        assert.deepEqual(trackEvent.firstCall.args[1], {
            // eslint-disable-next-line camelcase
            recommended_collection_id: QUnit.config.current.testId,
        });

        assert.equal(instance._preloadThumbnail.callCount, 1);
    });
});
