import React from 'react';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import assign from 'lodash/assign';
import sinon from 'sinon';

import {
    TRANSITION_IN_CLASS,
    TRANSITION_OUT_CLASS,
    SHOW_ON_HOVER_CLASS,
    ShowOnHoverContainer,
} from 'ui/containers/common/show-on-hover';

const DEFAULT_ARGS = Object.freeze({
    children: [<div id="testDiv" key="first" />,
        <div id="testDiv2" key="second" />],
    onHoverElement: () => {},
    onHoverOffElement: () => {},
    showUI: false,
});

function renderShowOnHover(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    return shallow(<ShowOnHoverContainer {...args} />);
}

reactTest('ui | containers | common | show on hover', function(hooks) {
    hooks.beforeEach(function() {
        this.onHoverElement = sinon.spy();
        this.onHoverOffElement = sinon.spy();
        this.component = renderShowOnHover({
            onHoverElement: this.onHoverElement,
            onHoverOffElement: this.onHoverOffElement,
        });
    });

    QUnit.test('renders with correct class', function(assert) {
        assert.equal(
            this.component.hasClass(SHOW_ON_HOVER_CLASS),
            true,
            'has SHOW_ON_HOVER_CLASS class'
        );
    });

    QUnit.test('renders with callbacks', function(assert) {
        assert.equal(
            this.component.props('onMouseEnter').hasOwnProperty('onMouseEnter'),
            true,
            'has an onMouseEnter prop'
        );
        assert.equal(
            this.component.props('onMouseLeave').hasOwnProperty('onMouseLeave'),
            true,
            'has an onMouseLeave prop'
        );
    });

    QUnit.test('renders with passed in children in correct order', function(assert) {
        assert.equal(
            this.component.children().length,
            DEFAULT_ARGS.children.length,
            'should have the correct number of children'
        );
        const firstChild = this.component.childAt(0);
        const secondChild = this.component.childAt(1);
        assert.equal(
            firstChild.matchesElement(DEFAULT_ARGS.children[0]),
            true,
            'should have the correct first child'
        );
        assert.equal(
            secondChild.matchesElement(DEFAULT_ARGS.children[1]),
            true,
            'should have the correct second child'
        );
    });

    QUnit.test('calls onHoverElement on mouseEnter', function(assert) {
        this.component.simulate('mouseEnter');
        assert.equal(
            this.onHoverElement.callCount,
            1,
            'onHoverElementshould have been called exactly once'
        );
    });

    QUnit.test('calls onHoverOffElement on mouseLeave', function(assert) {
        this.component.simulate('mouseLeave');
        assert.equal(
            this.onHoverOffElement.callCount,
            1,
            'onHoverOffElement should have been called exactly once'
        );
    });

    QUnit.module('componentWillReceiveProps', function() {
        QUnit.test('renders with additional TRANSITION_IN_CLASS when showUI is flipped to true', function(assert) {
            this.component.setProps({ showUI: true });
            assert.equal(
                this.component.hasClass(SHOW_ON_HOVER_CLASS),
                true,
                'has SHOW_ON_HOVER_CLASS class'
            );
            assert.equal(
                this.component.hasClass(TRANSITION_IN_CLASS),
                true,
                'should have TRANSITION_IN_CLASS'
            );
        });

        QUnit.test('renders with additional TRANSITION_OUT_CLASS when showUI is flipped to false', function(assert) {
            this.component = renderShowOnHover({
                showUI: true,
                onHoverElement: this.onHoverElement,
                onHoverOffElement: this.onHoverOffElement,
            });
            this.component.setProps({ showUI: false });
            assert.equal(
                this.component.hasClass(SHOW_ON_HOVER_CLASS),
                true,
                'has SHOW_ON_HOVER_CLASS class'
            );
            assert.equal(
                this.component.hasClass(TRANSITION_OUT_CLASS),
                true,
                'should have TRANSITION_OUT_CLASS'
            );
        });
    });
});
