import React from 'react';
import { reactTest } from 'tests/utils/react-test';
import assign from 'lodash/assign';
import sinon from 'sinon';
import I18Next from 'i18next';
import { Provider } from 'react-redux';
import { I18nextProvider } from 'react-i18next';
import { init as initStore } from 'state';
import { shallow, mount } from 'enzyme';
import { ExtensionSettingsContainer, EXTENSION_SETTINGS_TOOLTIP } from 'ui/containers/extension-settings';
import { ExtensionSettingsButton } from 'ui/components/extensions/extension-settings-button';
import { ExtensionsMenuManager } from 'ui/components/extensions/extensions-menu-manager';
import * as UIActions from 'actions/ui';
import { createExtensionParsed } from 'tests/fixtures/extensions';
import { buildFakeWindow } from 'tests/fakes/window.fake';
import { EXTENSION_MENU_MAIN } from 'util/extensions';

const DEFAULT_ARGS = Object.freeze({
    isLoggedIn: true,
    isSettingsShown: false,
    extensionsSubmenu: EXTENSION_MENU_MAIN,
    onShowSettings() {},
    onIdentityToggle() {},
    onMenuTransition() {},
    playerHeight: 0,
    trackEvent() {},
    windowObj: buildFakeWindow(),
    extensions: createExtensionParsed(),
    shouldEnableExtensions: true,
});

function renderSettings(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const container = <ExtensionSettingsContainer {...args} />;
    return shallow(container);
}

function mountRenderSettings(overrides = {}) {
    const store = initStore();
    const args = assign({}, DEFAULT_ARGS, overrides);
    const container = (
        <Provider store={store}>
            <I18nextProvider i18n={I18Next}>
                <ExtensionSettingsContainer {...args} />
            </I18nextProvider>
        </Provider>
    );
    return mount(container);
}

reactTest('ui | containers | extension-settings', function() {
    QUnit.test('returns a div', function(assert) {
        const container = renderSettings();
        assert.equal(container.type(), 'div');
    });

    QUnit.test('if shouldEnableExtensions is false, does not return a div', function(assert) {
        const container = renderSettings({
            shouldEnableExtensions: false,
        });
        assert.equal(container.type(), null);
    });

    QUnit.test('contains a SettingsButton', function(assert) {
        const container = renderSettings();
        const instance = container.instance();
        assert.ok(container.containsMatchingElement(
            <ExtensionSettingsButton
                onClick={instance.handleSettingsButtonClick}
                tooltip={EXTENSION_SETTINGS_TOOLTIP}
            />
        ));
    });

    QUnit.test('if no extensions are returned, render nothing', function(assert) {
        const container = renderSettings({ extensions: [] });
        assert.equal(container.type(), null);
    });

    QUnit.test('if only hidden extensions are returned, render nothing', function(assert) {
        const extensions = createExtensionParsed();
        extensions[0].anchor = 'hidden';
        const container = renderSettings({ extensions });
        assert.equal(container.type(), null);
    });

    QUnit.test('if isSettingsShown is false, does not contain a ExtensionsMenuManager', function(assert) {
        const container = renderSettings({
            isSettingsShown: false,
        });
        assert.notOk(container.containsMatchingElement(<ExtensionsMenuManager />));
    });

    QUnit.test('if isSettingsShown is true, show a ExtensionsMenuManager', function(assert) {
        const container = renderSettings({
            isSettingsShown: true,
        });
        const instance = container.instance();
        assert.ok(container.containsMatchingElement(
            <ExtensionsMenuManager
                extensions={DEFAULT_ARGS.extensions}
                isLoggedIn={DEFAULT_ARGS.isLoggedIn}
                onIdentityToggle={instance.handleIdentityToggleClick}
                playerHeight={DEFAULT_ARGS.playerHeight}
                windowObj={DEFAULT_ARGS.windowObj}
            />));
    });

    QUnit.module('on extension settings button click', function() {
        QUnit.test('extensions settings button invokes onShowSettings prop with expected value', function(assert) {
            const showSettingsSpy = sinon.spy();
            const container = renderSettings({
                isSettingsShown: false,
                onShowSettings: showSettingsSpy,
            });

            container.find(ExtensionSettingsButton).props().onClick();

            assert.equal(showSettingsSpy.callCount, 1, 'onShowSettings invoked once');
            assert.equal(showSettingsSpy.firstCall.args[0], UIActions.MENU_STATE_EXTENSIONS, 'show extensions menu');

            showSettingsSpy.reset();
            container.setProps({
                isSettingsShown: true,
            });
            container.find(ExtensionSettingsButton).props().onClick();

            assert.equal(showSettingsSpy.callCount, 1, 'onShowSettings invoked once');
            assert.equal(showSettingsSpy.firstCall.args[0], UIActions.MENU_STATE_NONE,
                'hide settings menu when it is shown');
        });
    });

    QUnit.test('handleClickOutside hides the settings menu when invoked', function(assert) {
        const showSettingsSpy = sinon.spy();
        const container = renderSettings({
            isSettingsShown: true,
            onShowSettings: showSettingsSpy,
        });

        container.instance().handleClickOutside();
        assert.equal(showSettingsSpy.callCount, 1, 'showSettings prop invoked');
        assert.equal(showSettingsSpy.firstCall.args[0], UIActions.MENU_STATE_NONE,
            'showSettings value is MENU_STATE_NONE to hide menu');
    });

    QUnit.test('handleMenuTransition calls the onMenuTransition prop function', function(assert) {
        const onMenuTransitionSpy = sinon.spy();
        const component = renderSettings({
            isSettingsShown: true,
            onMenuTransition: onMenuTransitionSpy,
        });

        const instance = component.instance();
        instance.handleMenuTransition(EXTENSION_MENU_MAIN);
        assert.ok(onMenuTransitionSpy.calledWith(EXTENSION_MENU_MAIN));
    });

    QUnit.test('handleMenuTransition does not transition to invalid menus', function(assert) {
        const onMenuTransitionSpy = sinon.spy();
        const component = renderSettings({
            isSettingsShown: true,
            onMenuTransition: onMenuTransitionSpy,
        });
        const instance = component.instance();
        const fakeIdentifier = 'fakemenuidentifier';

        instance.handleMenuTransition(fakeIdentifier);
        assert.equal(onMenuTransitionSpy.callCount, 0);
    });

    QUnit.test('on mount, add a blur handler to hide the menu', function(assert) {
        const windowObj = {
            addEventListener: sinon.spy(),
            removeEventListener: sinon.spy(),
        };
        const showSettingsSpy = sinon.spy();
        const container = mountRenderSettings({
            isSettingsShown: true,
            onShowSettings: showSettingsSpy,
            windowObj,
        });

        assert.equal(windowObj.addEventListener.callCount, 1, 'event listener added');
        assert.equal(windowObj.addEventListener.firstCall.args[0], 'blur', 'listener added for blur event');
        assert.equal(windowObj.removeEventListener.callCount, 0, 'no events removed');

        assert.equal(showSettingsSpy.callCount, 0, 'showSettings has not been invoked');

        windowObj.addEventListener.firstCall.args[1]();

        assert.equal(showSettingsSpy.callCount, 1, 'showSettings has been invoked');
        assert.equal(showSettingsSpy.firstCall.args[0], UIActions.MENU_STATE_NONE, 'invoked to hide the menu');

        container.unmount();

        assert.equal(windowObj.removeEventListener.callCount, 1, 'event listener removed');
        assert.equal(windowObj.removeEventListener.firstCall.args[0], 'blur', 'listener removed for blur event');
        assert.equal(
            windowObj.removeEventListener.firstCall.args[1],
            windowObj.addEventListener.firstCall.args[1],
            'same listener removed as added'
        );
    });
});
