import React from 'react';
import assign from 'lodash/assign';
import { reactTest } from 'tests/utils/react-test';
import { mount } from 'enzyme';
import { ExtensionsConfirmationContainer } from 'ui/containers/extensions-confirmation';
import sinon from 'sinon';

export const PRODUCT_PAYLOAD = Object.freeze({
    description: 'mock description',
    developerName: 'mock extension name',
    displayPrice: '$2.00',
    extensionName: 'Mock Component',
    shortDescription: 'mock short description',
    sku: 'mock-product-sku',
    title: 'mock-title',
});

const DEFAULT_ARGS = Object.freeze({
    purchaseIntent: null,
    clearPurchaseIntent() {},
});

function render(overrides = {}, method = mount) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <ExtensionsConfirmationContainer {...args} />;
    return method(component);
}

reactTest('ui | containers | extension-confirmation', function() {
    QUnit.test('renders nothing with no purchaseIntent', function(assert) {
        const component = render();
        assert.equal(component.find('div').length, 0, 'no content present');
    });

    QUnit.test('renders with purchaseIntent', function(assert) {
        const component = render({ purchaseIntent: {
            payload: PRODUCT_PAYLOAD,
            isLoggedIn: false,
            makePurchase() {},
        } });
        assert.ok(component.find('div').length > 0, 'content present');
    });

    QUnit.test('when logged in buy is clickable and calls makePurchase', function(assert) {
        const makePurchase = sinon.spy();
        const clearPurchaseIntent = sinon.spy();
        const component = render({
            clearPurchaseIntent,
            isLoggedIn: true,
            purchaseIntent: {
                payload: PRODUCT_PAYLOAD,
                makePurchase,
            },
        });

        assert.ok(component.find('div').length > 0, 'content present');
        assert.equal(makePurchase.callCount, 0, 'precond - no makePurchase without click');
        assert.equal(clearPurchaseIntent.callCount, 0, 'precond - no purchaseIntent change without click');
        component.find('[data-test-extension-confirmation-purchase]').simulate('click');

        assert.equal(makePurchase.callCount, 1, 'buying triggers makePurchase');
        assert.equal(clearPurchaseIntent.firstCall.args[0], null, 'buying triggers clearPurchaseIntent');
    });

    QUnit.test('when logged in cancel is clickable and clears UI', function(assert) {
        const makePurchase = sinon.spy();
        const clearPurchaseIntent = sinon.spy();
        const component = render({
            clearPurchaseIntent,
            isLoggedIn: true,
            purchaseIntent: {
                payload: PRODUCT_PAYLOAD,
                makePurchase,
            },
        });

        assert.ok(component.find('div').length > 0, 'content present');
        assert.equal(makePurchase.callCount, 0, 'precond - no makePurchase without click');
        assert.equal(clearPurchaseIntent.callCount, 0, 'precond - no purchaseIntent change without click');
        component.find('[data-test-extension-confirmation-cancel]').simulate('click');

        assert.equal(makePurchase.callCount, 0, 'does not trigger makePurchase');
        assert.equal(clearPurchaseIntent.firstCall.args[0], null, 'cancellation triggers clearPurchaseIntent');
    });

    QUnit.test('when logged out close is clickable and clears UI', function(assert) {
        const makePurchase = sinon.spy();
        const clearPurchaseIntent = sinon.spy();
        const component = render({
            clearPurchaseIntent,
            isLoggedIn: false,
            purchaseIntent: {
                payload: PRODUCT_PAYLOAD,
                makePurchase,
            },
        });

        assert.ok(component.find('div').length > 0, 'content present');
        assert.equal(makePurchase.callCount, 0, 'precond - no makePurchase without click');
        assert.equal(clearPurchaseIntent.callCount, 0, 'precond - no purchaseIntent change without click');
        component.find('[data-test-close-popup-button-close]').simulate('click');

        assert.equal(makePurchase.callCount, 0, 'does not trigger makePurchase');
        assert.equal(clearPurchaseIntent.firstCall.args[0], null, 'cancellation triggers clearPurchaseIntent');
    });
});
