import React from 'react';
import assign from 'lodash/assign';
import omit from 'lodash/omit';
import isFunction from 'lodash/isFunction';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { ExtensionOverlay } from 'ui/components/extensions/extension-overlay';
import { LiveTwitchContentStream } from 'stream/twitch-live';
import { VODTwitchContentStream } from 'stream/twitch-vod';
import { AdContentTypes } from 'actions/ads';
import { nullContentStream } from 'stream/null';
import * as PlayerType from 'util/player-type';
import { PLATFORM_WEB, PLATFORM_MOBILE_WEB } from 'state/env';
import { ONLINE_STATUS } from 'state/online-status';
import { TYPE_WATCH_PARTY } from 'state/stream-metadata';
import sinon from 'sinon';
import { toggleFullScreen } from 'actions/screen-mode';
import {
    EXTENSIONS_LOADED,
    updateExtensionIdentityLinking,
    setExtensionModal,
} from 'actions/extensions';
import { showExtensionsSubmenu } from 'actions/ui';
import { EXTENSION_MENU_MANAGE_ACCESS } from 'util/extensions';
import {
    ExtensionsContainer,
    ExtensionsContainerComponent,
    mapDispatchToProps,
} from 'ui/containers/extensions-container';
import { createExtensionParsed } from 'tests/fixtures/extensions';
import { FunctionAction } from 'extension-coordinator/lib/constants';

const DEFAULT_ARGS = Object.freeze({
    extensions: [],
    game: 'LUL of Legends',
    shouldEnableExtensions: true,
    shouldShowExtensions: true,
    width: 320,
    height: 180,
    language: 'zh',
    locale: 'zh-CN',
    loginId: 111111,
    videoResolution: '1920 x 1080',
    isLoggedIn: true,
    isInvisible: false,
    onIdentityLinked() {},
    onModalRequested() {},
    onIdShareRequested() {},
});

function renderExtensionsContainer(...args) {
    const props = assign.apply(null, [{}, DEFAULT_ARGS, {
        login: `user_${QUnit.config.current.testId}`,
        channel: `channel_${QUnit.config.current.testId}`,
        channelId: parseInt(QUnit.config.current.testId, 36),
        deviceId: `device_${QUnit.config.current.testId}`,
        platform: PLATFORM_WEB,
        playerType: 'popout',
        trackEvent() {},
        onExtensionDoubleClick() {},
        onIdentityToggle() {},
        onIdentityLinked() {},
    }].concat(args));

    return {
        props,
        component: shallow(<ExtensionsContainerComponent {...props} />),
    };
}

reactTest('ui | containers | extensions-container', function() {
    QUnit.module('ExtensionsContainerComponent', function() {
        QUnit.module('when there are no extensions', function(hooks) {
            hooks.beforeEach(function() {
                this.overrides = {
                    extensions: [],
                };
            });

            QUnit.test('should render nothing', function(assert) {
                const { component } = renderExtensionsContainer(this.overrides);

                assert.equal(component.type(), null);
            });
        });

        QUnit.module('when there are extensions', function(hooks) {
            hooks.beforeEach(function() {
                this.overrides = {
                    extensions: createExtensionParsed(),
                };
            });

            QUnit.test('should render a container with the set of extensions', function(assert) {
                const { props, component } = renderExtensionsContainer(this.overrides);
                const overlays = component.find(ExtensionOverlay);

                assert.equal(overlays.length, 1);

                const overlayProps = overlays.at(0).props();
                const callbacks = ['onDoubleClick', 'onIdentityLinked', 'onModalRequested'];
                const overlayPropsWithoutCallbacks = omit(overlayProps, callbacks);
                assert.deepEqual(overlayPropsWithoutCallbacks, {
                    width: DEFAULT_ARGS.width,
                    height: DEFAULT_ARGS.height,
                    extension: this.overrides.extensions[0],
                    game: DEFAULT_ARGS.game,
                    isLoggedIn: true,
                    language: DEFAULT_ARGS.language,
                    locale: DEFAULT_ARGS.locale,
                    loginId: 111111,
                    trackingProperties: {
                        channel: props.channel,
                        channelId: props.channelId,
                        deviceId: props.deviceId,
                        login: props.login,
                        platform: props.platform,
                        playerType: props.playerType,
                    },
                    onBeginPurchase: overlayPropsWithoutCallbacks.onBeginPurchase,
                });

                callbacks.forEach(fn => assert.ok(isFunction(overlayProps[fn])));
                assert.equal(overlays.at(0).key(), this.overrides.extensions[0].id, 'has a unique `key`');
            });

            QUnit.test('should enforce the player aspect ratio when the player size is not 16:9', function(assert) {
                const { component } = renderExtensionsContainer(this.overrides, {
                    width: 800,
                    height: 600,
                });

                const extensionComponent = component.find(ExtensionOverlay).at(0);

                assert.equal(extensionComponent.prop('width'), 800, 'width stays the same');
                assert.equal(extensionComponent.prop('height'), 800 * 9 / 16, 'height to be adjusted');
            });

            QUnit.test('should enforce the content aspect ratio when the player size is 16:9', function(assert) {
                const { component } = renderExtensionsContainer(this.overrides, {
                    width: 1280,
                    height: 720,
                    videoResolution: '800 x 600',
                });

                const extensionComponent = component.find(ExtensionOverlay).at(0);

                assert.equal(extensionComponent.prop('width'), 1280 * 3 / 4, 'width to be adjusted');
                assert.equal(extensionComponent.prop('height'), 720, 'height stays the same');
            });

            QUnit.test('mapDispatchToProps dispatches expected actions', function(assert) {
                const dispatch = sinon.spy();
                const mappedDispatches = mapDispatchToProps(dispatch);
                const extension = this.overrides.extensions[0];
                const modalRequest = 'twitch-ext-follow-action';

                mappedDispatches.onExtensionDoubleClick();
                assert.equal(dispatch.callCount, 1, 'one action dispatched');
                assert.deepEqual(dispatch.firstCall.args[0], toggleFullScreen(),
                    'toggle fullscreen action dispatched');

                dispatch.reset();

                mappedDispatches.onIdentityLinked(extension.id, true);
                assert.equal(dispatch.callCount, 1, 'one action dispatched');
                assert.deepEqual(dispatch.firstCall.args[0], updateExtensionIdentityLinking(extension.id, true),
                    'request extension identity toggle action dispatched with the correct ID');

                dispatch.reset();

                mappedDispatches.onModalRequested(modalRequest);
                assert.equal(dispatch.callCount, 1, 'one action dispatched');
                assert.deepEqual(dispatch.firstCall.args[0], setExtensionModal(modalRequest),
                    'modal request action is dispatched');

                dispatch.reset();

                mappedDispatches.onIdShareRequested();
                assert.equal(dispatch.callCount, 1, 'one action dispatched');
                assert.deepEqual(dispatch.firstCall.args[0], showExtensionsSubmenu(EXTENSION_MENU_MANAGE_ACCESS),
                    'showExtensionsSubmenu action is dispatched with EXTENSION_MENU_MANAGE_ACCESS');
            });

            QUnit.test('should not render overlays when shouldEnableExtensions is false', function(assert) {
                const { component } = renderExtensionsContainer(this.overrides, {
                    shouldEnableExtensions: false,
                });
                const overlays = component.find(ExtensionOverlay);

                assert.equal(overlays.length, 0, 'overlays are not rendered');
            });

            QUnit.test('should hide the host container when shouldShowExtensions is false', function(assert) {
                const { component } = renderExtensionsContainer(this.overrides, {
                    shouldShowExtensions: false,
                });
                const overlays = component.find(ExtensionOverlay);

                assert.ok(overlays.length > 0, 'overlays are still rendered');
                assert.ok(component.hasClass('invisible'), 'host container is invisible');
            });

            QUnit.test('should hide the host container when extension anchor is hidden', function(assert) {
                this.overrides.extensions[0].anchor = 'hidden';
                const { component } = renderExtensionsContainer(this.overrides, {});
                const overlays = component.find(ExtensionOverlay);

                assert.ok(overlays.length > 0, 'overlays are still rendered');
                assert.ok(component.hasClass('hide'), 'host container is hidden');
            });
        });
    });

    QUnit.module('_onModalRequested', function() {
        QUnit.test('when called with an ordinary modal request', function(assert) {
            const onModalRequestedSpy = sinon.spy();
            const { component } = renderExtensionsContainer({
                onModalRequested: onModalRequestedSpy,
            });

            const modalRequest = { action: FunctionAction.FollowAction };
            component.instance()._onModalRequested(modalRequest);

            assert.ok(
                onModalRequestedSpy.calledWith(modalRequest),
                'calls the passed onModalRequested prop with the request'
            );
        });

        QUnit.test('when called with an IdShareRequest', function(assert) {
            const onIdShareRequestedSpy = sinon.spy();
            const { component } = renderExtensionsContainer({
                onIdShareRequested: onIdShareRequestedSpy,
            });

            const modalRequest = { action: FunctionAction.IdShareRequest };
            component.instance()._onModalRequested(modalRequest);

            assert.equal(
                onIdShareRequestedSpy.callCount,
                1,
                'calls the passed onIdShareRequested prop with the request'
            );
        });
    });

    QUnit.module('extensions-container', function(hooks) {
        hooks.beforeEach(function() {
            this.state = {
                ads: {
                    currentMetadata: {
                        contentType: AdContentTypes.NONE,
                    },
                },
                analyticsTracker: {
                    trackEvent() {},
                },
                playback: {
                    contentShowing: true,
                    paused: false,
                },
                playerDimensions: {
                    width: 800,
                    height: 600,
                },
                env: {
                    platform: PLATFORM_WEB,
                    playerType: 'site',
                },
                extensions: {
                    channel: `channel-${QUnit.config.current.testId}`,
                    extensions: [],
                    loadingState: EXTENSIONS_LOADED,
                },
                onlineStatus: ONLINE_STATUS,
                lang: {
                    langCode: 'de-CH',
                },
                stream: nullContentStream,
                streamMetadata: {
                    channel: {
                        id: parseInt(QUnit.config.current.testId, 36),
                        name: `act-channel-${QUnit.config.current.testId}`,
                    },
                    channelName: `exp-channel-${QUnit.config.current.testId}`,
                    game: 'LUL of Legends',
                    streamType: '',
                },
                stats: {
                    videoStats: {
                        videoResolution: '1920 x 1080',
                    },
                },
                ui: {
                    isMini: false,
                },
                user: {
                    id: parseInt(QUnit.config.current.testId, 36),
                    name: `user_${QUnit.config.current.testId}`,
                },
            };

            this.store = {
                getState: () => this.state,
                dispatch: () => {},
                subscribe: () => {},
            };
        });

        QUnit.module('when the stream is a Live stream', function(hooks) {
            hooks.beforeEach(function() {
                this.state.stream = new LiveTwitchContentStream();
                this.state.onlineStatus = ONLINE_STATUS;
            });

            QUnit.test('should render and show extensions', function(assert) {
                const container = shallow(<ExtensionsContainer store={this.store} />);
                const component = container.find(ExtensionsContainerComponent);

                assert.equal(component.prop('shouldEnableExtensions'), true,
                    'extension iframes are running');
                assert.equal(component.prop('shouldShowExtensions'), true,
                    'extension iframes are visible');
            });

            QUnit.module('when the content is not showing', function(hooks) {
                hooks.beforeEach(function() {
                    this.state.playback.contentShowing = false;
                });

                QUnit.test('should not render extensions', function(assert) {
                    const container = shallow(<ExtensionsContainer store={this.store} />);
                    const component = container.find(ExtensionsContainerComponent);

                    assert.equal(component.prop('shouldEnableExtensions'), false);
                });
            });

            QUnit.module('when ads are playing', function(hooks) {
                hooks.beforeEach(function() {
                    this.state.ads.currentMetadata.contentType = AdContentTypes.STITCHED;
                });

                QUnit.test('extensions should render, but not be visible', function(assert) {
                    const container = shallow(<ExtensionsContainer store={this.store} />);
                    const component = container.find(ExtensionsContainerComponent);

                    assert.equal(component.prop('shouldEnableExtensions'), true,
                        'extension iframes are rendered');
                    assert.equal(component.prop('shouldShowExtensions'), false,
                        'no extension iframes are visible');
                });
            });

            QUnit.module('when in mini player', function(hooks) {
                hooks.beforeEach(function() {
                    this.state.ui.isMini = true;
                });

                QUnit.test('extensions should not be visible, but should be rendered', function(assert) {
                    const container = shallow(<ExtensionsContainer store={this.store} />);
                    const component = container.find(ExtensionsContainerComponent);

                    assert.equal(component.prop('shouldEnableExtensions'), true,
                        'extension iframes are running');
                    assert.equal(component.prop('shouldShowExtensions'), false,
                        'no extension iframes are visible');
                });
            });

            QUnit.module('when paused', function(hooks) {
                hooks.beforeEach(function() {
                    this.state.playback.paused = true;
                });

                QUnit.test('extensions should not be visible, but should be rendered', function(assert) {
                    const container = shallow(<ExtensionsContainer store={this.store} />);
                    const component = container.find(ExtensionsContainerComponent);

                    assert.equal(component.prop('shouldEnableExtensions'), true,
                        'extension iframes are running');
                    assert.equal(component.prop('shouldShowExtensions'), false,
                        'no extension iframes are visible');
                });
            });
        });

        QUnit.module('when the stream is a Live stream running a watch party', function(hooks) {
            hooks.beforeEach(function() {
                this.state.stream = new LiveTwitchContentStream();
                this.state.onlineStatus = ONLINE_STATUS;
                this.state.streamMetadata.streamType = TYPE_WATCH_PARTY;
            });

            QUnit.test('should not render extensions', function(assert) {
                const container = shallow(<ExtensionsContainer store={this.store} />);
                const component = container.find(ExtensionsContainerComponent);

                assert.equal(component.prop('shouldEnableExtensions'), false);
            });
        });

        QUnit.module('when the stream is a VOD stream', function(hooks) {
            hooks.beforeEach(function() {
                this.state.stream = new VODTwitchContentStream();
            });

            QUnit.test('should not render extensions', function(assert) {
                const container = shallow(<ExtensionsContainer store={this.store} />);
                const component = container.find(ExtensionsContainerComponent);

                assert.equal(component.prop('shouldEnableExtensions'), false);
            });
        });

        QUnit.module('when on a mobile device', function(hooks) {
            hooks.beforeEach(function() {
                this.state.stream = new LiveTwitchContentStream();
                this.state.env.platform = PLATFORM_MOBILE_WEB;
            });

            QUnit.test('should not render extensions', function(assert) {
                const container = shallow(<ExtensionsContainer store={this.store} />);
                const component = container.find(ExtensionsContainerComponent);

                assert.equal(component.prop('shouldEnableExtensions'), false);
            });
        });

        QUnit.test('it should not enable extensions for specific player types', function(assert) {
            const playerTypeEnablesExtensions = {
                [PlayerType.PLAYER_SITE]: true,
                [PlayerType.PLAYER_SITE_MINI]: true,
                [PlayerType.PLAYER_EMBED]: true,
                [PlayerType.PLAYER_POPOUT]: true,
                [PlayerType.PLAYER_FRONTPAGE]: false,
                [PlayerType.PLAYER_FEED]: false,
                [PlayerType.PLAYER_DASHBOARD]: true,
                [PlayerType.PLAYER_CREATIVE]: true,
                [PlayerType.PLAYER_CURSE]: false,
                [PlayerType.PLAYER_FACEBOOK]: false,
                [PlayerType.PLAYER_HIGHLIGHTER]: false,
                [PlayerType.PLAYER_TWITCH_EVERYWHERE]: false,
                [PlayerType.PLAYER_PULSE]: false,
                [PlayerType.PLAYER_AMAZON_VSE]: false,
            };

            this.state.stream = new LiveTwitchContentStream();
            Object.keys(playerTypeEnablesExtensions).forEach(playerType => {
                this.state.env.playerType = playerType;

                const shouldShow = playerTypeEnablesExtensions[playerType];
                const container = shallow(<ExtensionsContainer store={this.store} />);
                const component = container.find(ExtensionsContainerComponent);

                assert.equal(
                    component.prop('shouldEnableExtensions'),
                    playerTypeEnablesExtensions[playerType],
                    `should${shouldShow ? '' : ' not'} show extensions for ${playerType}`
                );
            });
        });
    });
});
