import React from 'react';
import sinon from 'sinon';
import merge from 'lodash/merge';
import { init as initStore } from 'state';
import { setWindow } from 'actions/window';
import { buildFakeWindow } from 'tests/fakes/window.fake';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { makeFollowPanel, mapStateToProps } from 'ui/containers/follow-panels/follow-panel-hoc';
import { NOTIFICATIONS_BUTTON_VIEW, NOTIFICATIONS_BUTTON_CLICK, FOLLOW_AND_NOTIFICATIONS_ON,
         NOTIFICATIONS_ON, NOTIFICATIONS_OFF } from 'actions/follow';
import { TWITCH_SIGNUP_URL } from 'settings';
import { CONTENT_SCREEN, VOD_RECOMMENDATION_SCREEN, ADVERTISEMENT_SCREEN } from 'actions/screen';

// eslint-disable-next-line react/prop-types, no-unused-vars
const MockFollowPanel = ({ channelName, notificationsEnabled, enable, disable, withInfo }) => {
    return <h1> Panel </h1>;
};

function renderFollowPanel(props) {
    const WrappedFollowPanel = makeFollowPanel(MockFollowPanel);
    const store = initStore();
    const fakeWindow = buildFakeWindow();
    store.dispatch(setWindow(fakeWindow));
    const component = <WrappedFollowPanel store={store} />;
    const unwrappedComponent = unwrapHOC(shallow(component));
    unwrappedComponent.setProps(props);
    return unwrappedComponent;
}

function unwrapHOC(component) {
    return shallow(component.instance().render());
}

reactTest('ui | containers | follow-panels', function() {
    QUnit.module('mapStateToProps', function() {
        // eslint-disable-next-line max-len
        QUnit.test('when follow.showFollowNotifcation is true, showFollowNotification is set to true', function(assert) {
            const modifiedState = merge(initStore().getState(), {
                follow: {
                    showFollowNotification: true,
                },
                ui: {
                    isMini: false,
                },
                playback: {
                    ended: true,
                },
            });
            const resultProps = mapStateToProps(modifiedState);
            assert.equal(resultProps.showFollowNotification, true);
        });

        // eslint-disable-next-line max-len
        QUnit.test('when follow.showFollowNotifcation is false, showFollowNotification is set to false', function(assert) {
            const modifiedState = merge(initStore().getState(), {
                follow: {
                    showFollowNotification: false,
                },
                ui: {
                    isMini: false,
                },
                playback: {
                    ended: true,
                },
            });
            const resultProps = mapStateToProps(modifiedState);
            assert.equal(resultProps.showFollowNotification, false);
        });

        QUnit.test('when ui.isMini, showFollowNotification is set to false', function(assert) {
            const modifiedState = merge(initStore().getState(), {
                follow: {
                    showFollowNotification: true,
                },
                ui: {
                    isMini: true,
                },
                playback: {
                    ended: true,
                },
            });
            const resultProps = mapStateToProps(modifiedState);
            assert.equal(resultProps.showFollowNotification, false);
        });

        QUnit.test('when playback.ended is false, set showFollowNotification to false', function(assert) {
            const modifiedState = merge(initStore().getState(), {
                follow: {
                    showFollowNotification: true,
                },
                ui: {
                    isMini: false,
                },
                playback: {
                    ended: false,
                },
            });
            const resultProps = mapStateToProps(modifiedState);
            assert.equal(resultProps.showFollowNotification, false);
        });

        // eslint-disable-next-line max-len
        QUnit.test('when screen is not content or vod recommendation, set showFollowNotification to false', function(assert) {
            const modifiedState = merge(initStore().getState(), {
                follow: {
                    showFollowNotification: true,
                },
                ui: {
                    isMini: false,
                },
                playback: {
                    ended: true,
                },
                screen: [ADVERTISEMENT_SCREEN],
            });
            const resultProps = mapStateToProps(modifiedState);
            assert.equal(resultProps.showFollowNotification, false);
        });

        // eslint-disable-next-line max-len
        QUnit.test('when screen is content screen, set showFollowNotification to true', function(assert) {
            const modifiedState = merge(initStore().getState(), {
                follow: {
                    showFollowNotification: true,
                },
                ui: {
                    isMini: false,
                },
                playback: {
                    ended: true,
                },
                screen: [CONTENT_SCREEN],
            });
            const resultProps = mapStateToProps(modifiedState);
            assert.equal(resultProps.showFollowNotification, true);
        });

        // eslint-disable-next-line max-len
        QUnit.test('when screen is vod recommendation screen, set showFollowNotification to true', function(assert) {
            const modifiedState = merge(initStore().getState(), {
                follow: {
                    showFollowNotification: true,
                },
                ui: {
                    isMini: false,
                },
                playback: {
                    ended: true,
                },
                screen: [VOD_RECOMMENDATION_SCREEN],
            });
            const resultProps = mapStateToProps(modifiedState);
            assert.equal(resultProps.showFollowNotification, true);
        });
    });

    QUnit.test('when showFollowNotification is false, render null', function(assert) {
        const component = renderFollowPanel({ showFollowNotification: false });
        assert.equal(component.type(), null);
    });

    QUnit.test('when showFollowNotification is true, renders a panel', function(assert) {
        const component = renderFollowPanel({ showFollowNotification: true });
        assert.equal(component.type(), MockFollowPanel);
    });

    // eslint-disable-next-line max-len
    QUnit.test('when showFollowNotification changes from false to true, it fires an analytics event', function(assert) {
        const trackEvent = sinon.spy();
        const component = renderFollowPanel({
            showFollowNotification: false,
            analytics: { trackEvent },
        });

        assert.equal(trackEvent.callCount, 0);

        component.setProps({ showFollowNotification: true });

        assert.equal(trackEvent.callCount, 1);
        assert.equal(trackEvent.firstCall.args[0], NOTIFICATIONS_BUTTON_VIEW);
    });

    // eslint-disable-next-line max-len
    QUnit.test('if user is not logged in, and on site, the enableNotifications method will invoke promptLoginModal', function(assert) {
        const promptLoginModal = sinon.spy();
        const component = renderFollowPanel({
            user: {
                id: 1,
                loggedIn: false,
            },
            channel: {
                id: 2,
                name: QUnit.config.current.testId,
            },
            promptLoginModal,
            playerType: 'site',
        });
        component.instance().enableNotifications();

        assert.equal(promptLoginModal.callCount, 1);
        assert.equal(promptLoginModal.firstCall.args[0], QUnit.config.current.testId);
    });

    // eslint-disable-next-line max-len
    QUnit.test('if user is not logged in, and not site, the enableNotifications method will use window.open', function(assert) {
        const open = sinon.spy();
        const component = renderFollowPanel({
            user: {
                id: 1,
                loggedIn: false,
            },
            channel: {
                id: 2,
                name: QUnit.config.current.testId,
            },
            win: { open },
        });
        component.instance().enableNotifications();

        assert.equal(open.callCount, 1);
        assert.equal(open.firstCall.args[0], TWITCH_SIGNUP_URL);
        assert.equal(open.firstCall.args[1], '_blank');
    });

    QUnit.test('if user is logged in, enableNotifications will be called with correct args', function(assert) {
        const enableNotifications = sinon.spy();
        const component = renderFollowPanel({
            user: {
                id: 1,
                loggedIn: true,
            },
            channel: {
                id: 2,
                name: QUnit.config.current.testId,
            },
            enableNotifications,
        });
        component.instance().enableNotifications();

        assert.equal(enableNotifications.callCount, 1);
        assert.equal(enableNotifications.firstCall.args[0], 1);
        assert.equal(enableNotifications.firstCall.args[1], 2);
    });

    QUnit.test('enableNotifications will track a click event, with the correct action', function(assert) {
        const trackEvent = sinon.spy();

        // When user is not following, and no notifications
        const component = renderFollowPanel({
            analytics: { trackEvent },
        });

        component.instance().enableNotifications();

        assert.equal(trackEvent.firstCall.args[0], NOTIFICATIONS_BUTTON_CLICK);
        assert.equal(trackEvent.firstCall.args[1].action, FOLLOW_AND_NOTIFICATIONS_ON);

        trackEvent.reset();

        // When user is following, but no notifications
        component.setProps({
            relationship: {
                following: true,
                notificationsEnabled: false,
            },
        });
        component.instance().enableNotifications();

        assert.equal(trackEvent.firstCall.args[0], NOTIFICATIONS_BUTTON_CLICK);
        assert.equal(trackEvent.firstCall.args[1].action, NOTIFICATIONS_ON);
    });

    QUnit.test('disableNotifications will track a click event, with the correct action', function(assert) {
        const trackEvent = sinon.spy();

        // When user is not following, and no notifications
        const component = renderFollowPanel({
            analytics: { trackEvent },
        });

        component.instance().disableNotifications();

        assert.equal(trackEvent.firstCall.args[0], NOTIFICATIONS_BUTTON_CLICK);
        assert.equal(trackEvent.firstCall.args[1].action, NOTIFICATIONS_OFF);
    });

    QUnit.test('if user is logged in, disableNotifications will be called with correct args', function(assert) {
        const disableNotifications = sinon.spy();
        const component = renderFollowPanel({
            user: {
                id: 1,
                loggedIn: true,
            },
            channel: {
                id: 2,
                name: QUnit.config.current.testId,
            },
            disableNotifications,
        });
        component.instance().disableNotifications();

        assert.equal(disableNotifications.callCount, 1);
        assert.equal(disableNotifications.firstCall.args[0], 1);
        assert.equal(disableNotifications.firstCall.args[1], 2);
    });
});
