import React from 'react';
import assign from 'lodash/assign';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { localStore } from 'tests/fakes/local-store.fake';
import sinon from 'sinon';
import { AgeGateOverlayContainer, KEY_AGE_GATE, AGE_GATE_CHANNEL_AGES } from 'ui/containers/overlays/age-gate-overlay';
import { AgeGateOverlay } from 'ui/components/overlays/age-gate-overlay';

const DEFAULT_ARGS = Object.freeze({
    playback: {
        autoplay: false,
    },
    streamMetadata: {
        channel: {
            name: 'monstercat',
        },
    },
    mutePlayer() {},
    contentIsShowing() {},
    play() {},
    pause() {},
    requestAds() {},
});

function renderAgeGateOverlay(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <AgeGateOverlayContainer {...args} />;
    return shallow(component);
}

export const TEST_AGE_GATE_CHANNEL_AGES =  Object.freeze({
    budlight: 21,
    budlightcanada: 19,
});

const MATURE_STREAM_METADATA = {
    channel: {
        name: 'budlight',
    },
};

const NON_MATURE_STREAM_METADATA = {
    channel: {
        name: 'monstercat',
    },
};

reactTest('ui | containers | overlays | age-gate-overlay', function(hooks) {
    hooks.beforeEach(function() {
        sinon.stub(localStore, 'set');
        sinon.stub(localStore, 'get').returns({});
    });

    hooks.afterEach(function() {
        localStore.set.restore();
        localStore.get.restore();
    });

    QUnit.test('on mount renders null', function(assert) {
        const component = renderAgeGateOverlay();
        assert.equal(component.type(), null);
    });

    // eslint-disable-next-line max-len
    QUnit.test('show overlay when stream metadata changes to mature stream that has not passed age gate', function(assert) {
        const component = renderAgeGateOverlay({});

        assert.equal(component.type(), null);

        localStore.get.withArgs(KEY_AGE_GATE, {}).returns({});
        component.setProps({ streamMetadata: MATURE_STREAM_METADATA });

        assert.equal(component.type(), AgeGateOverlay);
    });

    QUnit.test('do not show overlay when stream metadata changes to nonmature stream', function(assert) {
        const component = renderAgeGateOverlay({});
        localStore.get.withArgs(KEY_AGE_GATE, {}).returns({});
        component.setProps({ streamMetadata: MATURE_STREAM_METADATA });

        assert.equal(component.type(), AgeGateOverlay);

        component.setProps({ streamMetadata: NON_MATURE_STREAM_METADATA });

        assert.equal(component.type(), null);
    });

    // eslint-disable-next-line max-len
    QUnit.test('do not show overlay when when stream metadata changes to a mature stream that already passed age gate', function(assert) {
        const component = renderAgeGateOverlay({});
        localStore.get.withArgs(KEY_AGE_GATE, {}).returns({});
        component.setProps({ streamMetadata: MATURE_STREAM_METADATA });

        assert.equal(component.type(), AgeGateOverlay);

        localStore.get.withArgs(KEY_AGE_GATE, {}).returns({
            [MATURE_STREAM_METADATA.channel.name]: true,
        });
        component.setProps({ streamMetadata: MATURE_STREAM_METADATA });

        assert.equal(component.type(), null);
    });

    QUnit.test('calls mutePlayer, pause, and contentIsShowing prop when age gate overlay is shown', function(assert) {
        const mutePlayerSpy = sinon.spy();
        const pauseSpy = sinon.spy();
        const contentIsShowingSpy = sinon.spy();
        const component = renderAgeGateOverlay({
            mutePlayer: mutePlayerSpy,
            pause: pauseSpy,
            contentIsShowing: contentIsShowingSpy,
        });

        assert.equal(component.type(), null);

        mutePlayerSpy.reset();
        pauseSpy.reset();
        contentIsShowingSpy.reset();

        component.setProps({ streamMetadata: MATURE_STREAM_METADATA });

        assert.equal(component.type(), AgeGateOverlay);
        assert.equal(mutePlayerSpy.callCount, 1, 'muteplayer called once.');
        assert.equal(pauseSpy.callCount, 1, 'pause called once.');
        assert.equal(contentIsShowingSpy.callCount, 1, 'contentIsShowing called once.');
    });

    // eslint-disable-next-line max-len
    QUnit.test('age gate pass handler on the overlay calls mutePlayer, pause, and contentIsShowing prop and hides overlay', function(assert) {
        const mutePlayerSpy = sinon.spy();
        const playSpy = sinon.spy();
        const requestAdsSpy = sinon.spy();
        const component = renderAgeGateOverlay({
            playback: { autoplay: true },
            mutePlayer: mutePlayerSpy,
            play: playSpy,
            requestAds: requestAdsSpy,
        });

        component.setProps({ streamMetadata: MATURE_STREAM_METADATA });
        assert.equal(component.type(), AgeGateOverlay);

        const ageGatePassHandler = component.prop('handleAgeGatePassed');

        mutePlayerSpy.reset();
        playSpy.reset();
        requestAdsSpy.reset();

        ageGatePassHandler();

        component.update();

        assert.equal(mutePlayerSpy.callCount, 1, 'muteplayer called once.');
        assert.equal(playSpy.callCount, 1, 'pause called once.');
        assert.equal(requestAdsSpy.callCount, 1, 'requestAds called once.');

        assert.equal(component.type(), null);
    });

    reactTest('renders overlay if the channel included in AGE_GATE_CHANNEL_AGES object', function(hooks) {
        hooks.beforeEach(function() {
            localStore.clear();
        });

        QUnit.test('do not show overlay when stream is mature and not an age gate required channel', function(assert) {
            const component = renderAgeGateOverlay({});
            const MATURE_STREAM_METADATA = {
                streamMetadata: {
                    channel: {
                        name: 'notbudlight',
                        mature: true,
                    },
                },
            };
            const shouldShowOverlay = component.instance()._shouldShowOverlay(MATURE_STREAM_METADATA);
            assert.equal(shouldShowOverlay, false, 'overlay is not shown.');
        });

        QUnit.test('show overlay when stream is mature and an age gate required channel', function(assert) {
            const component = renderAgeGateOverlay({});
            const MATURE_STREAM_METADATA = {
                streamMetadata: {
                    channel: {
                        name: 'budlight',
                        mature: true,
                    },
                },
            };
            const shouldShowOverlay = component.instance()._shouldShowOverlay(MATURE_STREAM_METADATA);
            assert.equal(shouldShowOverlay, true, 'overlay is  shown.');
        });
    });

    reactTest('ensure AGE_GATE_CHANNEL_AGES object is valid', function() {
        QUnit.test('should have correct age gated channels and associated age requirements', function(assert) {
            assert.deepEqual(AGE_GATE_CHANNEL_AGES, TEST_AGE_GATE_CHANNEL_AGES, 'are equal');
        });
    });
});
