import React from 'react';
import assign from 'lodash/assign';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { ErrorOverlayContainer, mapStateToProps } from 'ui/containers/overlays/error-overlay';
import { ErrorOverlay as ErrorOverlayComponent } from 'ui/components/overlays/error-overlay';
import { CONTENT_SCREEN, VOD_RESTRICTED_SCREEN } from 'actions/screen';
import Errors from 'errors';

const DEFAULT_ARGS = Object.freeze({
    hasError: false,
    errorCode: 0,
    errorMessage: '',
    isVODRestricted: false,
});

function renderErrorOverlay(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <ErrorOverlayContainer {...args} />;
    return shallow(component);
}

reactTest('ui | containers | overlays | error-overlay', function() {
    QUnit.test('correctly maps state to props', function(assert) {
        const state = {
            error: {
                hasError: false,
                code: -1,
            },
            screen: [CONTENT_SCREEN],
        };

        assert.equal(
            mapStateToProps(state).hasError,
            state.error.hasError,
            'hasError is mapped correctly'
        );
        assert.equal(
            mapStateToProps(state).errorCode,
            state.error.code,
            'errorCode is mapped correctly'
        );
        assert.equal(
            mapStateToProps(state).isVODRestricted,
            false,
            'isVODRestricted is mapped correctly when screen[0] is not vod restricted'
        );
        assert.equal(
            mapStateToProps(state).errorMessage,
            '',
            'errorMessage mapped to empty string since hasError is false'
        );

        state.screen = [VOD_RESTRICTED_SCREEN];

        assert.deepEqual(
            mapStateToProps(state).isVODRestricted,
            true,
            'isVODRestricted is mapped correctly when screen[0] is vod restricted'
        );

        state.error.hasError = true;
        state.error.code = Errors.CODES.ABORTED;

        assert.equal(
            mapStateToProps(state).errorMessage,
            Errors.getMessage(state.error.code),
            'errorMessage mapped to correct error message since hasError is true'
        );
    });
    QUnit.test('renders null if hasError is false', function(assert) {
        const component = renderErrorOverlay();
        assert.equal(component.type(), null);
    });

    QUnit.test('renders a ErrorOverlayComponent if hasError', function(assert) {
        const component = renderErrorOverlay({ hasError: true });
        assert.equal(component.type(), ErrorOverlayComponent);
    });

    QUnit.test('passes the error code and message to the ErrorOverlayComponent', function(assert) {
        const errorCode = Errors.CODES.ABORTED;
        const errorMessage = Errors.getMessage(errorCode);
        const component = renderErrorOverlay({
            errorCode,
            errorMessage,
            hasError: true,
        });

        assert.equal(
            component.props().errorCode,
            errorCode,
            'passes correct error code'
        );
        assert.equal(
            component.props().errorMessage,
            errorMessage,
            'passes correct error message'
        );
    });
});
