import React from 'react';
import sinon from 'sinon';
import assign from 'lodash/assign';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { LoadingOverlayContainer, mapStateToProps, mapDispatchToProps } from 'ui/containers/overlays/loading-overlay';
import { LoadingOverlay as LoadingOverlayComponent } from 'ui/components/overlays/loading-overlay';
import { ADVERTISEMENT_SCREEN, CONTENT_SCREEN } from 'actions/screen';
import { CONNECTED, CONNECTING, UNAVAILABLE } from 'state/chromecast';
import { play } from 'actions/video-api';
import { buildFakeWindow } from 'tests/fakes/window.fake';

const DEFAULT_ARGS = Object.freeze({
    isCasting: false,
    isPaused: false,
    isLoading: false,
    isAdvertisementScreen: false,
    playStream() {},
    windowObj: buildFakeWindow(),
});

const DEFAULT_STORE_STATE = Object.freeze({
    playback: {
        paused: false,
        isLoading: false,
    },
    screen: [CONTENT_SCREEN],
    chromecast: {
        castingState: UNAVAILABLE,
    },
    window: {},
});

function renderLoadingOverlay(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <LoadingOverlayContainer {...args} />;
    return shallow(component);
}

reactTest('ui | containers | overlays | loading-overlay', function() {
    // eslint-disable-next-line max-len
    QUnit.test('renders LoadingOverlayComponent if isLoading, is not casting, is not paused, is not advertisment', function(assert) {
        const component = renderLoadingOverlay({
            isCasting: false,
            isPaused: false,
            isLoading: true,
            isAdvertisementScreen: false,
        });

        assert.equal(component.type(), LoadingOverlayComponent, 'is a loading overlay component');
    });

    function testNullRender(props) {
        QUnit.test('renders null if conditions not met', function(assert) {
            const msg = Object.keys(props).
                reduce((msg, propName) => `${msg}, ${propName}: ${props[propName]}`, 'Renders null when');

            const component = renderLoadingOverlay(props);
            assert.equal(component.type(), null, msg);
        });
    }

    testNullRender(assign({}, DEFAULT_ARGS, { isLoading: false }));
    testNullRender(assign({}, DEFAULT_ARGS, { isCasting: true }));
    testNullRender(assign({}, DEFAULT_ARGS, { isPaused: true }));
    testNullRender(assign({}, DEFAULT_ARGS, { isAdvertisementScreen: true }));

    QUnit.test('tells loading overlay to render offline overlay only if state.isUserOffline', function(assert) {
        const container = renderLoadingOverlay({
            isCasting: false,
            isPaused: false,
            isLoading: true,
            isAdvertisementScreen: false,
        });

        assert.equal(
            container.find(LoadingOverlayComponent).props().shouldShowOfflineOverlay,
            false,
            'does not show offline overlay by default');

        container.setState({
            isUserOffline: true,
        });

        assert.equal(
            container.find(LoadingOverlayComponent).props().shouldShowOfflineOverlay,
            true,
            'if state.isUserOffline, show offline overlay'
        );
    });

    QUnit.test('on mount, adds an `offline` event handler', function(assert) {
        const fakeWindow = buildFakeWindow();

        renderLoadingOverlay({
            windowObj: fakeWindow,
        });

        assert.equal(fakeWindow.addEventListener.calledWith('offline'), true, 'handler added');
    });

    QUnit.test('offline handler sets state.isUserOffline to true and records paused state', function(assert) {
        const fakeWindow = buildFakeWindow();
        const container = renderLoadingOverlay({
            isPaused: true,
            windowObj: fakeWindow,
        });

        assert.equal(container.state().isUserOffline, false, 'false by default');
        assert.equal(container.state().wasPausedOnOffline, false, 'false by default');

        fakeWindow.addEventListener.firstCall.args[1]();

        assert.equal(container.state().isUserOffline, true, 'set to true after handler');
        assert.equal(container.state().wasPausedOnOffline, true, 'set to true since it is paused');
    });

    QUnit.test('on mount, adds an `online` event handler', function(assert) {
        const fakeWindow = buildFakeWindow();

        renderLoadingOverlay({
            windowObj: fakeWindow,
        });

        assert.equal(fakeWindow.addEventListener.calledWith('online'), true, 'handler added');
    });

    QUnit.test('online event handler sets state.isUserOffline to false', function(assert) {
        const fakeWindow = buildFakeWindow();
        const container = renderLoadingOverlay({
            windowObj: fakeWindow,
        });

        container.setState({
            isUserOffline: true,
        });

        fakeWindow.addEventListener.secondCall.args[1]();

        assert.equal(container.state().isUserOffline, false, 'set to false after handler');
    });

    QUnit.test('online event handler calls `playStream` only if not paused when going offline', function(assert) {
        const fakeWindow = buildFakeWindow();
        const playStream = sinon.spy();
        const container = renderLoadingOverlay({
            playStream,
            windowObj: fakeWindow,
        });

        container.setState({
            wasPausedOnOffline: false,
        });

        fakeWindow.addEventListener.secondCall.args[1]();

        assert.equal(playStream.called, true, 'play stream called when not paused when going offline');

        playStream.reset();

        container.setState({
            wasPausedOnOffline: true,
        });

        fakeWindow.addEventListener.secondCall.args[1]();

        assert.equal(playStream.called, false, 'play stream not called when not paused going offline');
    });

    QUnit.module('mapStateToProps', function() {
        QUnit.test('sets isCasting to true when castingState is CONNECTED or CONNECTING', function(assert) {
            const propsWhenConnected = mapStateToProps(assign({}, DEFAULT_STORE_STATE, {
                chromecast: {
                    castingState: CONNECTED,
                },
            }));
            const propsWhenConnecting = mapStateToProps(assign({}, DEFAULT_STORE_STATE, {
                chromecast: {
                    castingState: CONNECTING,
                },
            }));

            assert.equal(propsWhenConnected.isCasting, true, 'isCasting is true when castingState is CONNECTED');
            assert.equal(propsWhenConnecting.isCasting, true, 'isCasting is true when castingState is CONNECTING');
        });

        QUnit.test('checks playback state to set props', function(assert) {
            const props = mapStateToProps(assign({}, DEFAULT_STORE_STATE, {
                playback: {
                    paused: true,
                    isLoading: true,
                },
            }));
            assert.equal(props.isPaused, true, 'checks paused state');
            assert.equal(props.isLoading, true, 'checks loading state');
        });

        QUnit.test('checks screen state to set props', function(assert) {
            const isAdProps = mapStateToProps(assign({}, DEFAULT_STORE_STATE, {
                screen: [ADVERTISEMENT_SCREEN],
            }));

            const isContentProps = mapStateToProps(assign({}, DEFAULT_STORE_STATE, {
                screen: [CONTENT_SCREEN],
            }));

            assert.equal(
                isAdProps.isAdvertisementScreen,
                true,
                'sets isAdvertisementScreen to true if screen is ADVERTISEMENT_SCREEN'
            );

            assert.equal(
                isContentProps.isAdvertisementScreen,
                false,
                'sets isAdvertisementScreen to false if screen is CONTENT_SCREEN'
            );
        });

        QUnit.test('windowObj is equivalent to state.window', function(assert) {
            const fakeWindow = { a: 'b' };
            const props = mapStateToProps(assign({}, DEFAULT_STORE_STATE, {
                window: fakeWindow,
            }));

            assert.deepEqual(props.windowObj, fakeWindow, 'window objects are the same');
        });
    });

    QUnit.module('mapDispatchToProps', function() {
        QUnit.test('playStream dispatches `play` action', function(assert) {
            const dispatch = sinon.spy();
            const mappedDispatches = mapDispatchToProps(dispatch);
            mappedDispatches.playStream();

            assert.equal(dispatch.calledWith(play()), true, 'play action dispatched');
        });
    });
});
