import React from 'react';
import assign from 'lodash/assign';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { localStore } from 'tests/fakes/local-store.fake';
import sinon from 'sinon';
import { MatureOverlayContainer, KEY_MATURE } from 'ui/containers/overlays/mature-overlay';
import { MatureOverlay } from 'ui/components/overlays/mature-overlay';
import { PLAYER_ANIMATED_THUMBNAILS } from 'util/player-type';

const DEFAULT_ARGS = Object.freeze({
    playback: {},
    playerType: '',
    streamMetadata: {},
    mutePlayer() {},
    contentIsShowing() {},
    play() {},
    pause() {},
    requestAds() {},
});

function renderMatureOverlay(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <MatureOverlayContainer {...args} />;
    return shallow(component);
}

reactTest('ui | containers | overlays | mature-overlay', function(hooks) {
    hooks.beforeEach(function() {
        localStore.clear();
    });

    QUnit.test('renders MatureOverlayContainer when showMatureOverlay is true ', function(assert) {
        const component = renderMatureOverlay();
        component.setState({
            showMatureOverlay: true,
        });

        assert.equal(component.type(), MatureOverlay);
    });

    QUnit.test('renders null when showMatureOverlay is false ', function(assert) {
        const component = renderMatureOverlay();
        component.setState({
            showMatureOverlay: false,
        });

        assert.equal(component.type(), null);
    });

    QUnit.test('do not show overlay when stream metadata changes to nonmature stream', function(assert) {
        const component = renderMatureOverlay();
        component.setProps({
            streamMetadata: {
                channel: {
                    mature: true,
                },
            },
        });
        assert.equal(component.type(), MatureOverlay);
        component.setProps({
            streamMetadata: {
                channel: {
                    mature: false,
                },
            },
        });
        assert.equal(component.type(), null);
    });

    QUnit.test('do not show overlay when mature key has been accepted', function(assert) {
        localStore.set(KEY_MATURE, true);
        const component = renderMatureOverlay();
        component.setProps({
            streamMetadata: {
                channel: {
                    mature: true,
                },
            },
        });
        assert.equal(component.type(), null);
    });

    QUnit.test('do not show overlay when playerType is animated_thumbnails', function(assert) {
        const component = renderMatureOverlay();
        component.setProps({
            playerType: PLAYER_ANIMATED_THUMBNAILS,
            streamMetadata: {
                channel: {
                    mature: true,
                },
            },
        });
        assert.equal(component.type(), MatureOverlay);

        component.setProps({
            playerType: PLAYER_ANIMATED_THUMBNAILS,
        });
        assert.equal(component.type(), null);
    });

    QUnit.test('player fires events when mature overlay is shown', function(assert) {
        const mutePlayerSpy = sinon.spy();
        const pauseSpy = sinon.spy();
        const contentIsShowingSpy = sinon.spy();
        const component = renderMatureOverlay({
            streamMetadata: {
                channel: {
                    name: 'test',
                    mature: true,
                },
            },
            mutePlayer: mutePlayerSpy,
            pause: pauseSpy,
            contentIsShowing: contentIsShowingSpy,
        });
        component.setState({
            showMatureOverlay: false,
        });
        component.setProps({
            streamMetadata: {
                channel: {
                    mature: true,
                },
            },
        });
        localStore.set(KEY_MATURE, false);

        assert.equal(mutePlayerSpy.callCount, 1, 'muteplayer called once.');
        assert.equal(pauseSpy.callCount, 1, 'pause called once.');
        assert.equal(contentIsShowingSpy.callCount, 1, 'contentIsShowing called once.');
        localStore.clear();
    });

    QUnit.test('player fires events when mature overlay is accepted', function(assert) {
        const mutePlayerSpy = sinon.spy();
        const playSpy = sinon.spy();
        const requestAdsSpy = sinon.spy();
        const component = renderMatureOverlay({
            playback: { autoplay: true },
            streamMetadata: {
                channel: {
                    name: 'test',
                    mature: true,
                },
            },
            mutePlayer: mutePlayerSpy,
            play: playSpy,
            requestAds: requestAdsSpy,
        });
        const instance = component.instance();

        instance.handleMatureAccepted();

        assert.equal(mutePlayerSpy.callCount, 1, 'muteplayer called once.');
        assert.equal(playSpy.callCount, 1, 'pause called once.');
        assert.equal(requestAdsSpy.callCount, 1, 'requestAds called once.');
    });
});
