import React from 'react';
import assign from 'lodash/assign';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { PlayButtonOverlayContainer, mapStateToProps } from 'ui/containers/overlays/play-button-overlay';
import { PlayButtonOverlay as PlayButtonOverlayComponent } from 'ui/components/overlays/play-button-overlay';
import { OFFLINE_STATUS, UNKNOWN_STATUS } from 'state/online-status';
import { CONTENT_MODE_VOD } from 'stream/twitch-vod';
import { CONTENT_MODE_LIVE } from 'stream/twitch-live';
import { VOD_RECOMMENDATION_SCREEN, ADVERTISEMENT_SCREEN, CONTENT_SCREEN } from 'actions/screen';
import { CONNECTING, CONNECTED, UNAVAILABLE } from 'state/chromecast';
import { SIDEBAR_VIEW, NO_COLLECTION_VIEW } from 'state/collection';

const DEFAULT_ARGS = Object.freeze({
    showOverlay: false,
    play() {},
});

function renderPlayButtonOverlay(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <PlayButtonOverlayContainer {...args} />;
    return shallow(component);
}

const DEFAULT_STATE = Object.freeze({
    chromecast: {
        castingState: UNAVAILABLE,
    },
    collection: {
        currentView: NO_COLLECTION_VIEW,
    },
    error: {
        hasError: false,
    },
    playback: {
        paused: true,
        ended: false,
    },
    onlineStatus: UNKNOWN_STATUS,
    screen: [CONTENT_SCREEN],
    stream: {
        contentType: CONTENT_MODE_LIVE,
    },
    ui: {
        isMini: false,
    },
});

reactTest('ui | containers | overlays | play-button-overlay', function() {
    QUnit.test('renders null if showOverlay is false', function(assert) {
        const component = renderPlayButtonOverlay();
        assert.equal(component.type(), null);
    });

    QUnit.test('renders PlayButtonOverlayComponent if showOverlay is true', function(assert) {
        const component = renderPlayButtonOverlay({ showOverlay: true });
        assert.equal(component.type(), PlayButtonOverlayComponent);
    });

    QUnit.test('passes play to overlay component', function(assert) {
        const component = renderPlayButtonOverlay({ showOverlay: true });
        assert.deepEqual(component.props().onClick, DEFAULT_ARGS.play, 'passes play method');
    });

    QUnit.test('passes collectionsSidebarOpen to child', function(assert) {
        const component = renderPlayButtonOverlay({
            showOverlay: true,
            collectionsSidebarOpen: true,
        });

        assert.deepEqual(
            component.props().collectionsSidebarOpen,
            true,
            'passes collectionsSidebarOpen'
        );
    });

    QUnit.module('mapStateToProps', function() {
        QUnit.test('sets collectionsSidebarOpen to true when sidebar is open', function(assert) {
            const state = assign({}, DEFAULT_STATE, {
                collection: {
                    currentView: SIDEBAR_VIEW,
                },
            });
            const props = mapStateToProps(state);
            assert.ok(props.collectionsSidebarOpen, 'should set collectionsSidebarOpen to true');
        });

        function testValidRenderConditions(subState, message) {
            QUnit.test('showOverlay renders true when conditions are met', function(assert) {
                const state = assign({}, DEFAULT_STATE, subState);
                const props = mapStateToProps(state);
                assert.ok(props.showOverlay, message);
            });
        }

        testValidRenderConditions({}, 'when paused and on live stream');
        testValidRenderConditions({
            stream: {
                contentType: CONTENT_MODE_VOD,
            },
            playback: {
                paused: false,
                ended: true,
            },
        }, 'when vod and ended');

        function testInvalidRenderConditions(subState, message) {
            QUnit.test('showOverlay renders false when conditions are not met', function(assert) {
                const state = assign({}, DEFAULT_STATE, subState);
                const props = mapStateToProps(state);
                assert.notOk(props.showOverlay, message);
            });
        }

        testInvalidRenderConditions({
            ui: {
                isMini: true,
            },
        }, 'hides when paused and isMini');

        testInvalidRenderConditions({
            error: {
                hasError: true,
            },
        }, 'hides when theres an error');

        testInvalidRenderConditions({
            playback: {
                paused: true,
                ended: true,
            },
            onlineStatus: OFFLINE_STATUS,
        }, 'hides when offline on livestream');

        testInvalidRenderConditions({
            screen: [VOD_RECOMMENDATION_SCREEN],
        }, 'hides when screen is vodrecommendations');

        testInvalidRenderConditions({
            screen: [ADVERTISEMENT_SCREEN],
        }, 'hides when screen is advertisement');

        testInvalidRenderConditions({
            chromecast: {
                castingState: CONNECTING,
            },
        }, 'hides when castingState is CONNECTING');

        testInvalidRenderConditions({
            chromecast: {
                castingState: CONNECTED,
            },
        }, 'hides when castingState is CONNECTED');
    });
});
