import React from 'react';
import assign from 'lodash/assign';
import { shallow } from 'enzyme';
import { reactTest } from 'tests/utils/react-test';
import { StormWarningContainer, mapStateToProps } from 'ui/containers/overlays/storm-warning';
import { StormWarningOverlay as StormWarningComponent } from 'ui/components/overlays/storm-warning';
import { STORM_WARNING_SCREEN, CONTENT_SCREEN } from 'actions/screen';

const DEFAULT_ARGS = Object.freeze({
    shouldRenderStormWarning: false,
});

const DEFAULT_STATE = Object.freeze({
    screen: [CONTENT_SCREEN],
});

function renderStormWarningOverlay(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <StormWarningContainer {...args} />;
    return shallow(component);
}

reactTest('ui | containers | overlays | storm-warning', function() {
    QUnit.test('renders null if shouldRenderStormWarning is false', function(assert) {
        const component = renderStormWarningOverlay();
        assert.equal(component.type(), null, 'storm-warning does not render on false');
    });

    QUnit.test('renders a StormWarningComponent if shouldRenderStormWarning is true', function(assert) {
        const component = renderStormWarningOverlay({
            shouldRenderStormWarning: true,
        });
        assert.equal(component.type(), StormWarningComponent, 'storm-warning renders on true');
    });

    reactTest('mapStateToProps', function() {
        QUnit.test('sets shouldRenderStormWarning to true when screens[0] === STORM_WARNING_SCREEN', function(assert) {
            const state = assign({}, DEFAULT_STATE, {
                screen: [STORM_WARNING_SCREEN, CONTENT_SCREEN],
            });
            const props = mapStateToProps(state);
            assert.equal(props.shouldRenderStormWarning, true, 'should set shouldRenderStormWarning to true');
        });

        QUnit.test('sets shouldRenderStormWarning to false when screens[0] !== STORM_WARNING_SCREEN', function(assert) {
            const state = assign({}, DEFAULT_STATE, {
                screen: [CONTENT_SCREEN],
            });
            const props = mapStateToProps(state);
            assert.equal(props.shouldRenderStormWarning, false, 'should set shouldRenderStormWarning to false');
        });
    });
});
