import React from 'react';
import assign from 'lodash/assign';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import sinon from 'sinon';
import { twitchHost } from 'settings';
import { SubscribeOverlayContainer } from 'ui/containers/overlays/subscribe-overlay';
import { SubscribeOverlay } from 'ui/components/overlays/subscribe-overlay';

const DEFAULT_ARGS = Object.freeze({
    screen: {},
    channelName: '',
    contentIsShowing() {},
    shouldRenderSubOverlay: false,
});

function renderSubscribeOverlay(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <SubscribeOverlayContainer {...args} />;
    return shallow(component);
}

reactTest('ui | containers | overlays | subscribe overlay', function() {
    QUnit.test('renders SubscribeOverlayContainer if shouldRenderSubOverlay is true', function(assert) {
        const component = renderSubscribeOverlay({
            shouldRenderSubOverlay: true,
        });

        assert.equal(component.type(), SubscribeOverlay, 'SubscribeOverlay renders.');
    });

    QUnit.test('renders null if shouldRenderSubOverlay is false', function(assert) {
        const component = renderSubscribeOverlay({
            shouldRenderSubOverlay: false,
        });

        assert.equal(component.type(), null, 'SubscribeOverlay does not render.');
    });

    QUnit.test('contentIsShowing is called when SubscribeOverlay is rendered', function(assert) {
        const contentIsShowing = sinon.spy();
        renderSubscribeOverlay({
            shouldRenderSubOverlay: true,
            contentIsShowing,
        });
        assert.equal(contentIsShowing.callCount, 1, 'contentIsShowing called once.');
    });

    QUnit.test('channelName is passed to component', function(assert) {
        const component = renderSubscribeOverlay({
            shouldRenderSubOverlay: true,
            channelName: 'day9tv',
        });
        assert.equal(component.props().channelName, 'day9tv', 'has expected channel name.');
    });

    QUnit.test('subscribeUrl is passed to component', function(assert) {
        const channelName = 'day9tv';
        const subscribeUrl = `${twitchHost}/${channelName}/subscribe?ref=chansub_overlay_subscribe`;

        const component = renderSubscribeOverlay({
            shouldRenderSubOverlay: true,
            channelName: 'day9tv',
            subscribeUrl: subscribeUrl,
        });
        assert.equal(component.props().subscribeUrl, subscribeUrl, 'has expected subscribe url.');
    });
});
