import React from 'react';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import assign from 'lodash/assign';
import sinon from 'sinon';

import { PlayerRootContainer } from 'ui/containers/player-root';

const DEFAULT_ARGS = Object.freeze({
    children: [<div id="testDiv" key="first" />,
        <div id="testDiv2" key="second" />],
    hideUI: () => {},
    paused: false,
    root: document.createElement('div'),
    showUIWithAutoHide: () => {},
});

function renderPlayerRoot(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    return shallow(<PlayerRootContainer {...args} />);
}

reactTest('ui | containers | player-root', function(hooks) {
    hooks.beforeEach(function() {
        this.showUIWithAutoHide = sinon.spy();
        this.hideUI = sinon.spy();
        this.component = renderPlayerRoot({
            showUIWithAutoHide: this.showUIWithAutoHide,
            hideUI: this.hideUI,
        });

        sinon.spy(this.component.instance().props.root, 'addEventListener');
        sinon.spy(this.component.instance().props.root, 'removeEventListener');
    });

    hooks.afterEach(function() {
        this.component.instance().props.root.addEventListener.restore();
        this.component.instance().props.root.removeEventListener.restore();
    });

    QUnit.test('renders with passed in children in correct order', function(assert) {
        assert.equal(
            this.component.children().length,
            DEFAULT_ARGS.children.length,
            'should have the correct number of children'
        );
        const firstChild = this.component.childAt(0);
        const secondChild = this.component.childAt(1);
        assert.equal(
            firstChild.matchesElement(DEFAULT_ARGS.children[0]),
            true,
            'should have the correct first child'
        );
        assert.equal(
            secondChild.matchesElement(DEFAULT_ARGS.children[1]),
            true,
            'should have the correct second child'
        );
    });

    QUnit.test('sets up callbacks on componentDidMount', function(assert) {
        this.component.instance().componentDidMount();
        assert.equal(
            this.component.instance().props.root.addEventListener.calledWith(
                'mouseenter',
                this.component.instance().showAndAutoHide
            ),
            true,
            'has expected mouseenter callback'
        );
        assert.equal(
            this.component.instance().props.root.addEventListener.calledWith(
                'mousemove',
                this.component.instance().showAndAutoHide
            ),
            true,
            'has expected mousemove callback'
        );
        assert.equal(
            this.component.instance().props.root.addEventListener.calledWith(
                'mouseleave',
                this.component.instance().hide
            ),
            true,
            'has expected mouseleave callback'
        );
    });

    QUnit.test('removes callbacks on componentWillUnmount', function(assert) {
        this.component.instance().componentWillUnmount();
        assert.equal(
            this.component.instance().props.root.removeEventListener.calledWith(
                'mouseenter',
                this.component.instance().showAndAutoHide
            ),
            true,
            'removes mouseenter callback'
        );
        assert.equal(
            this.component.instance().props.root.removeEventListener.calledWith(
                'mousemove',
                this.component.instance().showAndAutoHide
            ),
            true,
            'removes mousemove callback'
        );
        assert.equal(
            this.component.instance().props.root.removeEventListener.calledWith(
                'mouseleave',
                this.component.instance().hide
            ),
            true,
            'removes mouseleave callback'
        );
    });

    QUnit.module('onMouseEnter', function() {
        QUnit.test('calls showUIWithAutoHide if not paused', function(assert) {
            this.component.instance().props.root.dispatchEvent(new MouseEvent('mouseenter'));
            assert.equal(
                this.showUIWithAutoHide.callCount,
                1,
                'showUIWithAutoHide should have been called exactly once'
            );
        });

        QUnit.test('does not call showUIWithAutoHide if paused', function(assert) {
            this.component = renderPlayerRoot({
                paused: true,
                showUIWithAutoHide: this.showUIWithAutoHide,
                hideUI: this.hideUI,
            });
            this.component.simulate('mouseEnter');
            assert.equal(
                this.showUIWithAutoHide.callCount,
                0,
                'showUIWithAutoHide should not have been called'
            );
        });
    });

    QUnit.module('onMouseMove', function() {
        QUnit.test('calls showUIWithAutoHide if not paused', function(assert) {
            this.component.instance().props.root.dispatchEvent(new MouseEvent('mousemove'));
            assert.equal(
                this.showUIWithAutoHide.callCount,
                1,
                'showUIWithAutoHide should have been called exactly once'
            );
        });

        QUnit.test('does not call showUIWithAutoHide if paused', function(assert) {
            this.component = renderPlayerRoot({
                paused: true,
                showUIWithAutoHide: this.showUIWithAutoHide,
                hideUI: this.hideUI,
            });
            this.component.simulate('mouseMove');
            assert.equal(
                this.showUIWithAutoHide.callCount,
                0,
                'showUIWithAutoHide should not have been called'
            );
        });
    });

    QUnit.module('onMouseLeave', function() {
        QUnit.test('calls hideUI if not paused', function(assert) {
            this.component.instance().props.root.dispatchEvent(new MouseEvent('mouseleave'));
            assert.equal(this.hideUI.callCount, 1, 'hideUI should have been called exactly once');
        });

        QUnit.test('does not call hideUI if paused', function(assert) {
            this.component = renderPlayerRoot({
                paused: true,
                showUIWithAutoHide: this.showUIWithAutoHide,
                hideUI: this.hideUI,
            });
            this.component.simulate('mouseLeave');
            assert.equal(this.hideUI.callCount, 0, 'hideUI should not have been called');
        });
    });
});
