import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { OfflineRecommendations } from 'ui/components/recommendations/offline-recommendations';
import { NORMALIZED_V5_RECOMMENDED_VIDEOS } from 'tests/fixtures/recommendations';
import { VOD_RECOMMENDATION_SCREEN, CONTENT_SCREEN, ACTION_POP_SCREEN } from 'actions/screen';
import * as PlaybackActions from 'actions/playback';
import {
    OfflineRecommendationsContainer,
    mapStateToProps,
    mapDispatchToProps,
} from 'ui/containers/recommendations/offline-recommendations';
import {
    OFFLINE_RECOMMENDATIONS_TYPE,
    POST_VOD_RECOMMENDATIONS_TYPE,
    ACTION_CLEAR_RECOMMENDED_VODS,
} from 'actions/recommendations';

const DEFAULT_PROPS = Object.freeze({
    recommendedVideos: [NORMALIZED_V5_RECOMMENDED_VIDEOS],
    recommendationType: OFFLINE_RECOMMENDATIONS_TYPE,
    windowObj: {},
    showRecommendations: true,
    selectRecommendedVideo: () => {},
    popScreen: () =>  {},
    clearRecommendedVODs: () => {},
    i18n: {},
    playerDimensions: {
        height: 100,
        width: 100,
    },
    trackEvent: () => {},
    streamHasPlayed: false,
});

// a fake selectRecommendedVideo action
const FAKE_SELECT_REC_VIDEO_ACTION = 'a fake select recommended video action';

function renderOfflineRecommendations(overrides = {}) {
    const args = assign({}, DEFAULT_PROPS, overrides);
    const component = <OfflineRecommendationsContainer {...args} />;
    return shallow(component);
}

reactTest('ui | containers | recommendations | offline-recommendations', function(hooks) {
    hooks.beforeEach(function() {
        sinon.stub(PlaybackActions, 'selectRecommendedVideo', vodID => ({
            vodID: vodID,
            type: FAKE_SELECT_REC_VIDEO_ACTION,
        }));
    });

    hooks.afterEach(function() {
        PlaybackActions.selectRecommendedVideo.restore();
    });

    QUnit.test('correctly maps state to props', function(assert) {
        const state = {
            lang: {},
            screen: [CONTENT_SCREEN],
            recommendations: {
                videos: NORMALIZED_V5_RECOMMENDED_VIDEOS,
                type: OFFLINE_RECOMMENDATIONS_TYPE,
            },
            window: {},
            playerDimensions: {
                height: 100,
                width: 100,
            },
            analyticsTracker: {
                trackEvent: () => {},
            },
            playback: {
                hasPlayed: false,
                ended: false,
            },
        };

        assert.deepEqual(mapStateToProps(state), {
            recommendedVideos: state.recommendations.videos,
            recommendationType: state.recommendations.type,
            windowObj: state.window,
            showRecommendations: false,
            i18n: state.lang,
            playerDimensions: state.playerDimensions,
            trackEvent: state.analyticsTracker.trackEvent,
            streamHasPlayed: state.playback.hasPlayed,
        });

        state.screen = [VOD_RECOMMENDATION_SCREEN];
        state.playback.hasPlayed = true;
        state.playback.ended = true;

        assert.equal(mapStateToProps(state).showRecommendations, true);
        assert.equal(mapStateToProps(state).streamHasPlayed, state.playback.hasPlayed);

        // If stream has played and hasn't ended, don't show recommendations
        state.playback.hasPlayed = true;
        state.playback.ended = false;
        assert.equal(mapStateToProps(state).showRecommendations, false);
    });

    QUnit.test('correctly maps dispatches to props', function(assert) {
        const dispatchSpy = sinon.spy();
        const dispatchProps = mapDispatchToProps(dispatchSpy);
        const vodID = '1234567';

        dispatchProps.popScreen();
        assert.equal(dispatchSpy.firstCall.args[0].type, ACTION_POP_SCREEN);

        dispatchSpy.reset();

        dispatchProps.clearRecommendedVODs();
        assert.equal(dispatchSpy.firstCall.args[0].type, ACTION_CLEAR_RECOMMENDED_VODS);

        dispatchSpy.reset();

        dispatchProps.selectRecommendedVideo(vodID);
        assert.equal(dispatchSpy.firstCall.args[0].type, FAKE_SELECT_REC_VIDEO_ACTION);
        assert.equal(dispatchSpy.firstCall.args[0].vodID, vodID);
    });

    // eslint-disable-next-line max-len
    QUnit.test('show offline recs when showRecommendations is true and there are recommended videos and is correct recommendationType', function(assert) {
        const recommendationsComponent = renderOfflineRecommendations();

        assert.equal(recommendationsComponent.type(), OfflineRecommendations);
        assert.ok(recommendationsComponent.containsMatchingElement(
            <OfflineRecommendations
                videos={DEFAULT_PROPS.recommendedVideos}
                i18n={DEFAULT_PROPS.i18n}
                windowObj={DEFAULT_PROPS.windowObj}
                playerHeight={DEFAULT_PROPS.playerDimensions.height}
                playerWidth={DEFAULT_PROPS.playerDimensions.width}
            />
        ));
    });

    // eslint-disable-next-line max-len
    QUnit.test('onExit prop on offline recommendations pops the screen, and clears recommended videos', function(assert) {
        const clearRecommendedVODsSpy = sinon.spy();
        const popScreenSpy = sinon.spy();
        const recommendationsComponent = renderOfflineRecommendations({
            clearRecommendedVODs: clearRecommendedVODsSpy,
            popScreen: popScreenSpy,
        });
        const onExitProp = recommendationsComponent.prop('onExit');

        onExitProp();

        assert.equal(popScreenSpy.callCount, 1);
        assert.equal(clearRecommendedVODsSpy.callCount, 1);
    });

    // eslint-disable-next-line max-len
    QUnit.test('onSelectVOD prop on offline recommendations selects the vod, pops the screen, and clears recommended videos', function(assert) {
        const onSelectVODSpy = sinon.spy();
        const clearRecommendedVODsSpy = sinon.spy();
        const popScreenSpy = sinon.spy();
        const recommendationsComponent = renderOfflineRecommendations({
            clearRecommendedVODs: clearRecommendedVODsSpy,
            popScreen: popScreenSpy,
            selectRecommendedVideo: onSelectVODSpy,
        });
        const recommendedVideo = NORMALIZED_V5_RECOMMENDED_VIDEOS[0];

        const onSelectVODProp = recommendationsComponent.prop('onSelectVOD');

        onSelectVODProp(recommendedVideo);

        assert.equal(onSelectVODSpy.callCount, 1);
        assert.equal(onSelectVODSpy.firstCall.args[0], recommendedVideo.id);
        assert.equal(clearRecommendedVODsSpy.callCount, 1);
        assert.ok(clearRecommendedVODsSpy.calledAfter(onSelectVODSpy));
        assert.equal(popScreenSpy.callCount, 1);
        assert.ok(popScreenSpy.calledAfter(onSelectVODSpy));
    });

    QUnit.test('renders nothing if showRecommendations is false', function(assert) {
        const recommendationsComponent = renderOfflineRecommendations({
            showRecommendations: false,
        });

        assert.equal(recommendationsComponent.type(), null);
    });

    QUnit.test('renders nothing if there are no videos', function(assert) {
        const recommendationsComponent = renderOfflineRecommendations({
            recommendedVideos: [],
        });

        assert.equal(recommendationsComponent.type(), null);
    });

    QUnit.test('renders nothing if recommendationsType is not offline recs', function(assert) {
        const recommendationsComponent = renderOfflineRecommendations({
            recommendationType: POST_VOD_RECOMMENDATIONS_TYPE,
        });

        assert.equal(recommendationsComponent.type(), null);
    });
});

