import React from 'react';
import { reactTest } from 'tests/utils/react-test';
import assign from 'lodash/assign';
import sinon from 'sinon';
import I18Next from 'i18next';
import { Provider } from 'react-redux';
import { I18nextProvider } from 'react-i18next';
import { init as initStore } from 'state';
import { shallow, mount } from 'enzyme';
import { SettingsContainer, mapStateToProps } from 'ui/containers/settings';
import { SettingsBanner } from 'ui/containers/settings/settings-banner';
import { SettingsButton } from 'ui/components/settings/settings-button';
import { MenuManager } from 'ui/components/settings/menu-manager';
import { buildFakeWindow } from 'tests/fakes/window.fake';
import * as UIActions from 'actions/ui';

const DEFAULT_ARGS = Object.freeze({
    isAutoQuality: false,
    isEnded: false,
    isPaused: false,
    isSettingsShown: false,
    onShowSettings() {},
    playerHeight: 0,
    qualitiesAvailable: false,
    qualityIsChanging: false,
    trackEvent() {},
    windowObj: buildFakeWindow(),
});

function renderSettings(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const container = <SettingsContainer {...args} />;
    return shallow(container);
}

function mountRenderSettings(overrides = {}) {
    const store = initStore();
    const args = assign({}, DEFAULT_ARGS, overrides);
    const container = (
        <Provider store={store}>
            <I18nextProvider i18n={I18Next}>
                <SettingsContainer {...args} />
            </I18nextProvider>
        </Provider>
    );
    return mount(container);
}

reactTest('ui | containers | settings', function() {
    QUnit.test('returns a div', function(assert) {
        const container = renderSettings();
        assert.equal(container.type(), 'div');
    });

    QUnit.test('contains a SettingsBanner', function(assert) {
        const container = renderSettings();
        assert.ok(container.containsMatchingElement(<SettingsBanner />));
    });

    QUnit.test('contains a SettingsButton', function(assert) {
        const container = renderSettings();
        const instance = container.instance();
        assert.ok(container.containsMatchingElement(
            <SettingsButton
                onClick={instance.handleSettingsButtonClick}
            />
        ));
    });

    QUnit.test('if isSettingsShown is false, does not contain a MenuManager', function(assert) {
        const container = renderSettings({
            isSettingsShown: false,
        });
        assert.notOk(container.containsMatchingElement(<MenuManager />));
    });

    QUnit.test('if isSettingsShown is true, show a menuManager', function(assert) {
        const container = renderSettings({
            isSettingsShown: true,
        });
        assert.ok(container.containsMatchingElement(<MenuManager />));
    });

    QUnit.module('on settings button click', function() {
        QUnit.test('onClick of the settings button invokes onShowSettings prop with expected value', function(assert) {
            const showSettingsSpy = sinon.spy();
            const container = renderSettings({
                isSettingsShown: false,
                onShowSettings: showSettingsSpy,
            });

            container.find(SettingsButton).props().onClick();

            assert.equal(showSettingsSpy.callCount, 1, 'onShowSettings invoked once');
            assert.equal(showSettingsSpy.firstCall.args[0], UIActions.MENU_STATE_SETTINGS, 'show settings menu');

            showSettingsSpy.reset();
            container.setProps({
                isSettingsShown: true,
            });
            container.find(SettingsButton).props().onClick();

            assert.equal(showSettingsSpy.callCount, 1, 'onShowSettings invoked once');
            assert.equal(showSettingsSpy.firstCall.args[0], UIActions.MENU_STATE_NONE,
                'hide settings menu when it is shown');
        });

        QUnit.test('onClick invokes trackEvent prop with correct event', function(assert) {
            const trackEvent = sinon.spy();
            const container = renderSettings({
                trackEvent,
            });

            container.find(SettingsButton).props().onClick();

            assert.equal(trackEvent.callCount, 1, 'trackEvent invoked');
            assert.equal(trackEvent.firstCall.args[0], 'player_click_options', 'invoked with correct event name');
        });
    });

    QUnit.module('showSpinningAnimation', function(hooks) {
        hooks.beforeEach(function() {
            this.container = renderSettings({
                isAutoQuality: false,
                qualitiesAvailable: true,
                qualityIsChanging: true,
                isClipsPlayerType: false,
            });
        });
        QUnit.test('is true if not auto, qualities are available, and quality is changing', function(assert) {
            assert.ok(this.container.find(SettingsButton).props().showSpinningAnimation);
        });

        QUnit.test('is false if auto', function(assert) {
            this.container.setProps({
                isAutoQuality: true,
            });
            assert.notOk(this.container.find(SettingsButton).props().showSpinningAnimation);
        });

        QUnit.test('is false if quality is not changing', function(assert) {
            this.container.setProps({
                qualityIsChanging: false,
            });
            assert.notOk(this.container.find(SettingsButton).props().showSpinningAnimation);
        });

        QUnit.test('is false if qualities are not available', function(assert) {
            this.container.setProps({
                qualitiesAvailable: false,
            });
            assert.notOk(this.container.find(SettingsButton).props().showSpinningAnimation);
        });

        QUnit.test('is false if ended', function(assert) {
            this.container.setProps({
                isEnded: true,
            });
            assert.notOk(this.container.find(SettingsButton).props().showSpinningAnimation);
        });

        QUnit.test('is false if paused', function(assert) {
            this.container.setProps({
                isPaused: true,
            });
            assert.notOk(this.container.find(SettingsButton).props().showSpinningAnimation);
        });

        QUnit.test('is false if ClipsPlayerType', function(assert) {
            this.container.setProps({
                isClipsPlayerType: true,
            });
            assert.notOk(this.container.find(SettingsButton).props().showSpinningAnimation);
        });
    });

    QUnit.test('qualityIsChanging prop is set when selected quality is not current quality', function(assert) {
        const fakeQualityState = {
            selected: 'low',
            current: 'medium',
            available: [],
        };
        const fakeState = assign({}, initStore().getState(), {
            quality: fakeQualityState,
        });

        let stateProps = mapStateToProps(fakeState);
        assert.ok(stateProps.qualityIsChanging, 'is true when unequal');

        fakeQualityState.current = fakeQualityState.selected;
        stateProps = mapStateToProps(fakeState);
        assert.equal(stateProps.qualityIsChanging, false, 'is false when equal');
    });

    QUnit.test('handleClickOutside hides the settings menu when invoked', function(assert) {
        const showSettingsSpy = sinon.spy();
        const container = renderSettings({
            isSettingsShown: true,
            onShowSettings: showSettingsSpy,
        });

        container.instance().handleClickOutside();
        assert.equal(showSettingsSpy.callCount, 1, 'showSettings prop invoked');
        assert.equal(showSettingsSpy.firstCall.args[0], UIActions.MENU_STATE_NONE,
            'showSettings value is MENU_STATE_NONE to hide menu');
    });

    QUnit.test('on mount, add a blur handler to hide the menu', function(assert) {
        const windowObj = {
            addEventListener: sinon.spy(),
            removeEventListener: sinon.spy(),
        };
        const showSettingsSpy = sinon.spy();
        const container = mountRenderSettings({
            isSettingsShown: true,
            onShowSettings: showSettingsSpy,
            windowObj,
        });

        assert.equal(windowObj.addEventListener.callCount, 1, 'event listener added');
        assert.equal(windowObj.addEventListener.firstCall.args[0], 'blur', 'listener added for blur event');
        assert.equal(windowObj.removeEventListener.callCount, 0, 'no events removed');

        assert.equal(showSettingsSpy.callCount, 0, 'showSettings has not been invoked');

        windowObj.addEventListener.firstCall.args[1]();

        assert.equal(showSettingsSpy.callCount, 1, 'showSettings has been invoked');
        assert.equal(showSettingsSpy.firstCall.args[0], UIActions.MENU_STATE_NONE, 'invoked to hide the menu');

        container.unmount();

        assert.equal(windowObj.removeEventListener.callCount, 1, 'event listener removed');
        assert.equal(windowObj.removeEventListener.firstCall.args[0], 'blur', 'listener removed for blur event');
        assert.equal(
            windowObj.removeEventListener.firstCall.args[1],
            windowObj.addEventListener.firstCall.args[1],
            'same listener removed as added'
        );
    });
});
