import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { AdvancedMenuContainer, mapDispatchToProps, mapStateToProps } from 'ui/containers/settings/advanced-menu';
import { AdvancedMenu as AdvancedMenuComponent } from 'ui/components/settings/advanced-menu';
import { BACKEND_PLAYER_CORE } from 'backend/player-core';
import { BACKEND_MEDIA_PLAYER } from 'backend/mediaplayer';
import { BACKEND_FLASH } from 'backend/flash';
import { displayStats } from 'actions/stats';
import { ACTION_SET_PERSISTENCE } from 'actions/settings';
import { changeBackend } from 'actions/video-api';
import { Experiment } from 'ui/containers/common/experiments-container';
import { HTML5_TOGGLE_REMOVAL } from 'experiments';
import { init as initStore } from 'state';
import * as BackendUtils from 'backend/util';
import { PLAYER_SITE, PLAYER_CURSE } from 'util/player-type';

const DEFAULT_ARGS = Object.freeze({
    availableBackends: [],
    isCursePlayer: false,
    isHTML5Player: false,
    isMiniPlayerEnabled: false,
    isSitePlayer: false,
    onMenuTransition() {},
    statsDisplayed: false,
    toggleHTML5Player() {},
    toggleMiniPlayer() {},
    toggleVideoStats() {},
});

function renderAdvancedMenu(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <AdvancedMenuContainer {...args} />;
    return shallow(component);
}

reactTest('ui | containers | settings | advanced-menu', function(hooks) {
    hooks.beforeEach(function() {
        sinon.stub(BackendUtils, 'getAvailableBackends');
    });

    hooks.afterEach(function() {
        BackendUtils.getAvailableBackends.restore();
    });

    QUnit.test('correctly maps state to props', function(assert) {
        const expectedAvailableBackends = [BACKEND_MEDIA_PLAYER];
        const state = {
            backendInfo: {
                playerBackendType: BACKEND_PLAYER_CORE,
            },
            env: {
                playerType: PLAYER_CURSE,
            },
            settings: {
                persistenceEnabled: true,
            },
            stats: {
                displayed: true,
            },
        };

        BackendUtils.getAvailableBackends.withArgs(state.env.playerType).returns(expectedAvailableBackends);

        assert.deepEqual(mapStateToProps(state), {
            isCursePlayer: state.env.playerType === PLAYER_CURSE,
            isHTML5Player: (
                state.backendInfo.playerBackendType === BACKEND_PLAYER_CORE ||
                state.backendInfo.playerBackendType === BACKEND_MEDIA_PLAYER
            ),
            isMiniPlayerEnabled: state.settings.persistenceEnabled,
            isSitePlayer: state.env.playerType === PLAYER_SITE,
            statsDisplayed: state.stats.displayed,
            availableBackends: expectedAvailableBackends,
        });

        state.backendInfo.playerBackendType = BACKEND_MEDIA_PLAYER;
        assert.equal(mapStateToProps(state).isHTML5Player, true);
        state.backendInfo.playerBackendType = BACKEND_FLASH;
        assert.equal(mapStateToProps(state).isHTML5Player, false);
        state.env.playerType = PLAYER_SITE;
        assert.equal(mapStateToProps(state).isCursePlayer, false);
        assert.equal(mapStateToProps(state).isSitePlayer, true);
    });

    QUnit.test('returns a div', function(assert) {
        const container = renderAdvancedMenu();
        assert.equal(container.type(), 'div');
    });

    QUnit.test('the AdvancedMenuComponent has the expected args', function(assert) {
        const container = renderAdvancedMenu();
        const component = container.find(AdvancedMenuComponent).get(1);

        const expectedPropTypes = {
            html5PlayerEnabled: 'boolean',
            html5ToggleEnabled: 'boolean',
            isShowingVideoStats: 'boolean',
            miniPlayerEnabled: 'boolean',
            miniPlayerToggleEnabled: 'boolean',
            onMenuTransition: 'function',
            onMiniPlayerToggle: 'function',
            onStatsToggle: 'function',
            onHTML5Toggle: 'function',
        };

        Object.keys(expectedPropTypes).map(propName => {
            assert.equal(typeof component.props[propName], expectedPropTypes[propName]);
        });
    });

    // eslint-disable-next-line max-len
    QUnit.test('if not curse player, and can play player core, mediaplayer and flash, enable html5 toggle', function(assert) {
        const container = renderAdvancedMenu({
            availableBackends: [
                BACKEND_PLAYER_CORE,
                BACKEND_MEDIA_PLAYER,
                BACKEND_FLASH,
            ],
        });

        const component = container.find(AdvancedMenuComponent).get(0);

        assert.equal(component.props.html5ToggleEnabled, true, 'html5toggle enabled');
    });

    // eslint-disable-next-line max-len
    QUnit.test('if not curse player, and can only play player core, mediaplayer, do not enable html5 toggle', function(assert) {
        const container = renderAdvancedMenu({
            availableBackends: [
                BACKEND_PLAYER_CORE,
                BACKEND_MEDIA_PLAYER,
            ],
        });
        const component = container.find(AdvancedMenuComponent).get(0);

        assert.equal(component.props.html5ToggleEnabled, false, 'html5toggle not enabled');
    });

    QUnit.test('if not curse player, and can only play flash, do not enable html5 toggle', function(assert) {
        const container = renderAdvancedMenu({
            availableBackends: [BACKEND_FLASH],
        });
        const component = container.find(AdvancedMenuComponent).get(0);

        assert.equal(component.props.html5ToggleEnabled, false, 'html5toggle not enabled');
    });

    // eslint-disable-next-line max-len
    QUnit.test('if curse player, and can play player core, mediaplayer and flash, do not enable html5 toggle ', function(assert) {
        const container = renderAdvancedMenu({
            isCursePlayer: true,
            availableBackends: [
                BACKEND_PLAYER_CORE,
                BACKEND_MEDIA_PLAYER,
                BACKEND_FLASH,
            ],
        });

        assert.equal(container.find(AdvancedMenuComponent).get(0).props.html5ToggleEnabled, false);
    });

    QUnit.test('toggleVideoStats invokes prop with right value', function(assert) {
        let statsDisplayed = false;
        const toggleStatsSpy = sinon.spy();
        const container = renderAdvancedMenu({
            statsDisplayed,
            toggleVideoStats: toggleStatsSpy,
        });

        container.instance().toggleVideoStats();
        assert.equal(toggleStatsSpy.callCount, 1, 'prop invoked once');
        assert.equal(toggleStatsSpy.firstCall.args[0], !statsDisplayed, 'prop invoked to show stats');

        toggleStatsSpy.reset();
        statsDisplayed = true;
        container.setProps({
            statsDisplayed,
        });

        container.instance().toggleVideoStats();
        assert.equal(toggleStatsSpy.callCount, 1, 'prop invoked once');
        assert.equal(toggleStatsSpy.firstCall.args[0], !statsDisplayed, 'prop invoked to hide stats');
    });

    QUnit.test('toggleMiniPlayer invokes prop with right value', function(assert) {
        let isMiniPlayerEnabled = false;
        const toggleMiniPlayerSpy = sinon.spy();
        const container = renderAdvancedMenu({
            isMiniPlayerEnabled,
            toggleMiniPlayer: toggleMiniPlayerSpy,
        });

        container.instance().toggleMiniPlayer();
        assert.equal(toggleMiniPlayerSpy.callCount, 1, 'prop invoked once');
        assert.equal(toggleMiniPlayerSpy.firstCall.args[0], !isMiniPlayerEnabled, 'enables miniplayer');

        toggleMiniPlayerSpy.reset();
        isMiniPlayerEnabled = true;
        container.setProps({
            isMiniPlayerEnabled,
        });

        container.instance().toggleMiniPlayer();
        assert.equal(toggleMiniPlayerSpy.callCount, 1, 'prop invoked once');
        assert.equal(toggleMiniPlayerSpy.firstCall.args[0], !isMiniPlayerEnabled, 'disables miniplayer');
    });

    QUnit.test('toggleHTML5Player invokes prop with right value', function(assert) {
        let isHTML5Player = false;
        const toggleHTML5Spy = sinon.spy();
        const container = renderAdvancedMenu({
            isHTML5Player,
            toggleHTML5Player: toggleHTML5Spy,
        });

        container.instance().toggleHTML5Player();
        assert.equal(toggleHTML5Spy.callCount, 1, 'prop invoked once');
        assert.equal(toggleHTML5Spy.firstCall.args[0], !isHTML5Player, 'enables html5');

        toggleHTML5Spy.reset();
        isHTML5Player = true;
        container.setProps({
            isHTML5Player,
        });

        container.instance().toggleHTML5Player();
        assert.equal(toggleHTML5Spy.callCount, 1, 'prop invoked once');
        assert.equal(toggleHTML5Spy.firstCall.args[0], !isHTML5Player, 'disables html5');
    });

    QUnit.test('toggleVideoStats prop dispatches appropriate actions', function(assert) {
        const dispatchSpy = sinon.spy();
        const mappedDispatches = mapDispatchToProps(dispatchSpy);

        mappedDispatches.toggleVideoStats(true);
        assert.equal(dispatchSpy.callCount, 1, '1 actions dispatched');
        assert.deepEqual(dispatchSpy.firstCall.args[0], displayStats(true), 'display stats action dispatched');

        dispatchSpy.reset();

        mappedDispatches.toggleVideoStats(false);
        assert.equal(dispatchSpy.callCount, 1, '1 actions dispatched');
        assert.deepEqual(dispatchSpy.firstCall.args[0], displayStats(false), 'hide stats action dispatched');
    });

    QUnit.test('toggleMiniPlayer prop dispatches appropriate actions', function(assert) {
        const dispatchSpy = sinon.spy();
        const mappedDispatches = mapDispatchToProps(dispatchSpy);
        const fakeState = {
            analyticsTracker: {
                trackEvent() {},
            },
        };

        mappedDispatches.toggleMiniPlayer(true);
        assert.equal(dispatchSpy.callCount, 1, 'one action dispatched');

        let thunk = dispatchSpy.firstCall.args[0];
        thunk(dispatchSpy, () => fakeState);

        assert.equal(dispatchSpy.callCount, 2, 'second action dispatched thunk');
        assert.deepEqual(
            dispatchSpy.secondCall.args[0],
            {
                type: ACTION_SET_PERSISTENCE,
                value: true,
            },
            'enable miniplayer action dispatched from thunk'
        );

        dispatchSpy.reset();

        mappedDispatches.toggleMiniPlayer(false);
        assert.equal(dispatchSpy.callCount, 1, 'one action dispatched');

        thunk = dispatchSpy.firstCall.args[0];
        thunk(dispatchSpy, () => fakeState);

        assert.equal(dispatchSpy.callCount, 2, 'second action dispatched from thunk');
        assert.deepEqual(
            dispatchSpy.secondCall.args[0],
            {
                type: ACTION_SET_PERSISTENCE,
                value: false,
            },
            'disable miniplayer action dispatched from thunk'
        );
    });

    QUnit.test('toggleHTML5Player prop dispatches appropriate actions', function(assert) {
        const dispatchSpy = sinon.spy();
        const mappedDispatches = mapDispatchToProps(dispatchSpy);

        mappedDispatches.toggleHTML5Player(true);
        assert.equal(dispatchSpy.callCount, 1, 'one action dispatched');
        assert.deepEqual(dispatchSpy.firstCall.args[0], changeBackend(BACKEND_PLAYER_CORE), 'enable html5');

        dispatchSpy.reset();

        mappedDispatches.toggleHTML5Player(false);
        assert.equal(dispatchSpy.callCount, 1, 'one action dispatched');
        assert.deepEqual(dispatchSpy.firstCall.args[0], changeBackend(BACKEND_FLASH), 'enable flash');
    });

    QUnit.module('HTML5 Toggle Removal experiment', function() {
        QUnit.test('should have an experiment container for keep', function(assert) {
            const container = renderAdvancedMenu();
            const keepExperimentContainer = container.find(Experiment).get(0);
            assert.equal(keepExperimentContainer.props.uuid, HTML5_TOGGLE_REMOVAL, 'has correct uuid');
            assert.equal(keepExperimentContainer.props.renderValue, 'keep', 'has correct renderValue');
        });

        QUnit.test('should have an experiment container for remove', function(assert) {
            const container = renderAdvancedMenu();
            const removeExperimentContainer = container.find(Experiment).get(1);
            assert.equal(removeExperimentContainer.props.uuid, HTML5_TOGGLE_REMOVAL, 'has correct uuid');
            assert.equal(removeExperimentContainer.props.renderValue, 'remove', 'has correct renderValue');
        });
    });

    QUnit.module('mapStateToProps', function() {
        QUnit.test('isHTML5Player is true if type is player core or media player', function(assert) {
            const fakeState = assign({}, initStore().getState(), {
                backendInfo: {
                    playerBackendType: BACKEND_PLAYER_CORE,
                },
            });

            let mappedProps = mapStateToProps(fakeState);

            assert.equal(mappedProps.isHTML5Player, true, 'player core means html5 player');

            fakeState.backendInfo.playerBackendType = BACKEND_MEDIA_PLAYER;
            mappedProps = mapStateToProps(fakeState);

            assert.equal(mappedProps.isHTML5Player, true, 'mediaplayer means html5 player');
        });
    });
});
