import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { MainMenuContainer, mapDispatchToProps } from 'ui/containers/settings/main-menu';
import { MainMenu as MainMenuComponent } from 'ui/components/settings/main-menu';
import { toURLString as timestampUrlString } from 'util/timestamp';
import { pause } from 'actions/video-api';
import { showMenu, MENU_STATE_NONE } from 'actions/ui';
import { PLAYER_AMAZON_VSE, PLAYER_IMDB, PLAYER_SITE } from 'util/player-type';
import { showCCModal } from 'actions/captions';
import { parse as parseParams } from 'util/params';

const DEFAULT_ARGS = Object.freeze({
    availableQualities: [],
    captionsAvailable: false,
    collection: {},
    currentTime: 0,
    hideSettingsMenu() {},
    isVod: false,
    showCaptionsOptions() {},
    onMenuTransition() {},
    pausePlayback() {},
    playback: {},
    playerType: PLAYER_SITE,
    selectedQuality: '',
    stream: {},
    streamUrl: '',
    windowObj: {},
});

function renderMainMenu(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <MainMenuContainer {...args} />;
    return shallow(component);
}

reactTest('ui | containers | settings | main-menu', function() {
    QUnit.test('returns a MainMenuComponent', function(assert) {
        const container = renderMainMenu();
        assert.equal(container.type(), MainMenuComponent);
    });

    QUnit.test('onPopoutPlayer invokes pausePlayback prop, hides menu and opens a new window', function(assert) {
        const pauseSpy = sinon.spy();
        const hideSpy = sinon.spy();
        const fakeWindow = {
            open: sinon.spy(),
        };
        const container = renderMainMenu({
            hideSettingsMenu: hideSpy,
            pausePlayback: pauseSpy,
            windowObj: fakeWindow,
        });
        container.find(MainMenuComponent).props().onPopoutPlayer();
        assert.equal(pauseSpy.callCount, 1, 'playback paused');
        assert.equal(hideSpy.callCount, 1, 'settings menu hidden');
        assert.equal(fakeWindow.open.callCount, 1, 'new window opened');
        assert.ok(pauseSpy.calledBefore(fakeWindow.open), 'playback paused first');
    });

    QUnit.test('if playerType is IMDb, onPopoutPlayer opens a url with &player=imdb', function(assert) {
        const pauseSpy = sinon.spy();
        const hideSpy = sinon.spy();
        const fakeWindow = {
            open: sinon.spy(),
        };
        const container = renderMainMenu({
            hideSettingsMenu: hideSpy,
            pausePlayback: pauseSpy,
            windowObj: fakeWindow,
            playerType: PLAYER_IMDB,
        });
        container.find(MainMenuComponent).props().onPopoutPlayer();
        assert.equal(fakeWindow.open.callCount, 1, 'new window opened');

        const popoutUrl = fakeWindow.open.firstCall.args[0];
        const popoutParams = parseParams(String(popoutUrl.split('?')[1]));
        assert.equal(popoutParams.player, PLAYER_IMDB, 'adds playerType IMDB if current playerType is IMDB');
    });

    QUnit.test('if playerType is Amazon VSE, canPopoutPlayer is false', function(assert) {
        const container = renderMainMenu({
            playerType: PLAYER_AMAZON_VSE,
        });
        assert.equal(container.find(MainMenuComponent).props().canPopoutPlayer, false);
    });

    QUnit.test('onShowCaptionsOptions calls hideSettingsMenu and showCaptionsOptions', function(assert) {
        const fakeCCElement = {
            setAttribute: sinon.spy(),
        };
        const getElementStub = sinon.stub().returns(fakeCCElement);
        const fakeWindow = {
            document: {
                getElementById: getElementStub,
            },
        };

        const hideSettingsMenu = sinon.spy();
        const showCaptionsOptions = sinon.spy();

        const container = renderMainMenu({
            windowObj: fakeWindow,
            hideSettingsMenu,
            showCaptionsOptions,
        });

        container.find(MainMenuComponent).props().onShowCaptionsOptions();

        assert.equal(hideSettingsMenu.callCount, 1, 'hideSettingsMenu called once');
        assert.equal(showCaptionsOptions.callCount, 1, 'showCaptionsOptions called once');
    });

    QUnit.test('selected quality object name is passed to MainMenuComponent', function(assert) {
        const selectedQuality = 'low';
        const selectedQualityName = 'how low can you go';
        const availableQualities = [
            {
                group: selectedQuality,
                name: selectedQualityName,
            },
        ];

        const container = renderMainMenu({
            availableQualities,
            selectedQuality,
        });

        assert.equal(container.find(MainMenuComponent).props().selectedQualityName, selectedQualityName);
    });

    QUnit.test('vodUrlWithTimestamp passed to MainMenuComponent in constructed correctly', function(assert) {
        const streamUrl = 'http://twitch.tv/monstercat';
        const currentTime = 50;
        const container = renderMainMenu({
            currentTime,
            streamUrl,
        });

        const expectedUrl = `${streamUrl}?t=${timestampUrlString(currentTime)}`;
        assert.equal(container.find(MainMenuComponent).props().vodUrlWithTimestamp, expectedUrl);
    });

    QUnit.test('if no available qualities, shouldShowQualityMenu is false', function(assert) {
        const availableQualities = [];
        const container = renderMainMenu({
            availableQualities,
        });
        assert.notOk(container.find(MainMenuComponent).props().shouldShowQualityMenu);
    });

    QUnit.module('mapDispatchToProps', function() {
        QUnit.test('has pausePlayback prop which fires expected action', function(assert) {
            const dispatchSpy = sinon.spy();
            const dispatchProps = mapDispatchToProps(dispatchSpy);

            dispatchProps.pausePlayback();
            assert.equal(dispatchSpy.callCount, 1, 'one action dispatched');
            assert.deepEqual(dispatchSpy.firstCall.args[0], pause(), 'expected action dispatched');
        });

        QUnit.test('has hideSettingsMenu prop which fires expected action', function(assert) {
            const dispatchSpy = sinon.spy();
            const dispatchProps = mapDispatchToProps(dispatchSpy);

            dispatchProps.hideSettingsMenu();
            assert.equal(dispatchSpy.callCount, 1, 'one action dispatched');
            assert.deepEqual(dispatchSpy.firstCall.args[0], showMenu(MENU_STATE_NONE),
                'expected action dispatched');
        });

        QUnit.test('has showCaptionsOptions prop which fires expected action', function(assert) {
            const dispatchSpy = sinon.spy();
            const { showCaptionsOptions } = mapDispatchToProps(dispatchSpy);

            showCaptionsOptions();
            assert.equal(dispatchSpy.callCount, 1, 'one action dispatched');
            assert.deepEqual(dispatchSpy.firstCall.args[0], showCCModal(true),
                'expected action dispatched');
        });
    });
});
