import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { PlaybackSpeedMenuContainer, mapDispatchToProps } from 'ui/containers/settings/playback-speed-menu';
import { PlaybackSpeedMenu as PlaybackSpeedMenuComponent } from 'ui/components/settings/playback-speed-menu';
import { changePlaybackRate } from 'actions/video-api';
import { showMenu, MENU_STATE_NONE } from 'actions/ui';

const DEFAULT_ARGS = Object.freeze({
    currentSpeed: 1.0,
    onMenuTransition() {},
    selectNewPlaybackRate() {},
    tracker: {
        trackEvent() {},
    },
});

function renderContainer(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <PlaybackSpeedMenuContainer {...args} />;
    return shallow(component);
}

reactTest('ui | containers | settings | playback-speed-menu', function() {
    QUnit.test('returns a PlaybackSpeedMenuComponent', function(assert) {
        const component = renderContainer();
        assert.equal(component.type(), PlaybackSpeedMenuComponent);
    });

    QUnit.test('the PlaybackSpeedMenuComponent has the expected args', function(assert) {
        const currentSpeed = 1.0;
        const onMenuTransition = sinon.spy();
        const component = renderContainer({
            currentSpeed,
            onMenuTransition,
        });
        const props = component.find(PlaybackSpeedMenuComponent).props();

        assert.equal(props.currentSpeed, currentSpeed, 'current speed arg is passed in');
        assert.equal(props.onMenuTransition, onMenuTransition, 'menu transition arg is passed in');
    });

    QUnit.test('onSelectPlaybackSpeed prop invokes selectNewPlaybackRate dispatch prop', function(assert) {
        const spy = sinon.spy();
        const newSpeed = 2.0;
        const container = renderContainer({
            selectNewPlaybackRate: spy,
        });

        container.find(PlaybackSpeedMenuComponent).props().onSelectPlaybackSpeed(newSpeed);

        assert.equal(spy.callCount, 1, 'dispatch prop invoked once');
        assert.equal(spy.firstCall.args[0], newSpeed, 'invoked with speed arg');
    });

    QUnit.test('selectNewPlaybackRate dispatches the expected actions', function(assert) {
        const dispatchSpy = sinon.spy();
        const dispatchProps = mapDispatchToProps(dispatchSpy);
        const playbackRate = 2.0;

        dispatchProps.selectNewPlaybackRate(playbackRate);
        assert.equal(dispatchSpy.callCount, 2, 'two actions dispatched');
        assert.deepEqual(
            dispatchSpy.firstCall.args[0],
            changePlaybackRate(playbackRate),
            'changePlaybackRate dispatched'
        );
        assert.deepEqual(
            dispatchSpy.secondCall.args[0],
            showMenu(MENU_STATE_NONE),
            'hide settings menu action dispatched'
        );
    });

    QUnit.test('on selecting a new playback speed, the correct tracking event is fired', function(assert) {
        const fakeTracker = {
            trackEvent: sinon.spy(),
        };
        const newSpeed = 2.0;
        const container = renderContainer({
            tracker: fakeTracker,
        });

        container.find(PlaybackSpeedMenuComponent).props().onSelectPlaybackSpeed(newSpeed);
        assert.equal(fakeTracker.trackEvent.callCount, 1, 'one event was fired');
        assert.equal(fakeTracker.trackEvent.firstCall.args[0], 'player_click_playback', 'has correct event name');
        assert.deepEqual(fakeTracker.trackEvent.firstCall.args[1], { speed: newSpeed }, 'has correct properties');
    });
});
