import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { QualityMenuContainer, mapDispatchToProps } from 'ui/containers/settings/quality-menu';
import { QualityMenu as QualityMenuComponent } from 'ui/components/settings/quality-menu';
import { ACTION_SELECT_QUALITY } from 'actions/quality';
import { showMenu, MENU_STATE_NONE } from 'actions/ui';
import { trackEvent } from 'actions/analytics-tracker';

const DEFAULT_ARGS = Object.freeze({
    availableQualities: [],
    selectedQuality: '',
    onMenuTransition() {},
    handleSelectQuality() {},
});

function renderQualityMenu(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <QualityMenuContainer {...args} />;
    return shallow(component);
}

reactTest('ui | containers | settings | quality-menu', function() {
    QUnit.test('returns a QualityMenuComponent', function(assert) {
        const component = renderQualityMenu();
        assert.equal(component.type(), QualityMenuComponent);
    });

    QUnit.test('the QualityMenuComponent has expected args', function(assert) {
        const fakeAvailableQualities = [
            { name: 'fake quality' },
        ];
        const menuTransitionSpy = sinon.spy();
        const qualitySelectSpy = sinon.spy();
        const selectedQuality = 'low';
        const component = renderQualityMenu({
            availableQualities: fakeAvailableQualities,
            selectedQuality,
            onMenuTransition: menuTransitionSpy,
            handleSelectQuality: qualitySelectSpy,
        });

        const props = component.find(QualityMenuComponent).props();
        assert.deepEqual(props.availableQualities, fakeAvailableQualities, 'available qualities is passed in');
        assert.equal(props.selectedQuality, selectedQuality, 'selectedQuality passed in');
        assert.equal(props.onMenuTransition, menuTransitionSpy, 'menu transition handler passed in');
        assert.equal(props.onQualitySelect, qualitySelectSpy, 'quality select handler passed in');
    });

    // eslint-disable-next-line max-len
    QUnit.test('handleSelectQuality prop dispatches selectQuality, setPreferredQuality, hides menu, and tracks event', function(assert) {
        const dispatchSpy = sinon.spy();
        const fakeQualityObj = {
            name: 'fake quality',
            group: 'fake quality group',
        };
        const fakeState = {
            analyticsTracker: {
                trackEvent() {},
            },
            stream: {
                restrictedBitrates: '',
            },
            quality: {},
            manifestInfo: {},
        };

        mapDispatchToProps(dispatchSpy).handleSelectQuality(fakeQualityObj);

        assert.equal(dispatchSpy.callCount, 4, 'dispatch is called three times');
        assert.deepEqual(dispatchSpy.thirdCall.args[0], showMenu(MENU_STATE_NONE));
        assert.deepEqual(dispatchSpy.getCall(3).args[0], trackEvent('player_click_quality_change'));

        const selectQualityThunk = dispatchSpy.getCalls()[0].args[0];

        dispatchSpy.reset();

        selectQualityThunk(dispatchSpy, () => fakeState);

        assert.equal(dispatchSpy.callCount, 1, 'selectQuality thunk dispatches quality');
        assert.equal(dispatchSpy.firstCall.args[0].type, ACTION_SELECT_QUALITY, 'is the selectQuality action');
    });
});
