import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { ReportIssuesMenuContainer, mapDispatchToProps } from 'ui/containers/settings/report-issues-menu';
import { ReportIssuesMenu as ReportIssuesMenuComponent,
         REPORT_FEEDBACK_DELAY } from 'ui/components/settings/report-issues-menu';
import { submitVideoIssueReport } from 'actions/video-api';
import { showMenu, MENU_STATE_NONE } from 'actions/ui';

const DEFAULT_ARGS = Object.freeze({
    handleVideoIssueReport() {},
    onMenuTransition() {},
    windowObj: {
        setTimeout() {},
    },
});

function renderContainer(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const container = <ReportIssuesMenuContainer {...args} />;
    return shallow(container);
}

reactTest('ui | containers | settings | report-issues-menu', function() {
    QUnit.test('returns a ReportIssuesMenuComponent', function(assert) {
        const container = renderContainer();
        assert.equal(container.type(), ReportIssuesMenuComponent, 'has report issues menu component');
    });

    QUnit.test('handleVideoIssueReport has correct functionality', function(assert) {
        const issueReportSpy = sinon.spy();
        const settingsHideSpy = sinon.spy();
        const fakeWindow = {
            setTimeout: sinon.spy(),
        };
        const container = renderContainer({
            handleVideoIssueReport: issueReportSpy,
            hideSettingsMenu: settingsHideSpy,
            windowObj: fakeWindow,
        });
        const instance = container.instance();

        instance.handleVideoIssueReport();
        assert.equal(issueReportSpy.callCount, 1, 'handleVideoIssueReport prop invoked');
        assert.equal(fakeWindow.setTimeout.callCount, 1, 'timeout set');
        assert.equal(fakeWindow.setTimeout.firstCall.args[1], REPORT_FEEDBACK_DELAY, 'delay same as component');
        assert.equal(settingsHideSpy.callCount, 0, 'settings menu not hidden yet');
        fakeWindow.setTimeout.firstCall.args[0]();
        assert.equal(settingsHideSpy.callCount, 1, 'settings menu hidden after a delay');
    });

    QUnit.test('handleVideoIssueReport prop has correct dispatch', function(assert) {
        const spy = sinon.spy();
        const dispatchProps = mapDispatchToProps(spy);
        const fakeValue = 'too many ads';

        dispatchProps.handleVideoIssueReport(fakeValue);

        assert.equal(spy.callCount, 1, 'one action dispatched');
        assert.deepEqual(spy.firstCall.args[0], submitVideoIssueReport(fakeValue), 'is correct action');
    });

    QUnit.test('hideSettingsMenu prop has correct dispatch', function(assert) {
        const spy = sinon.spy();
        const dispatchProps = mapDispatchToProps(spy);

        dispatchProps.hideSettingsMenu();

        assert.equal(spy.callCount, 1, 'one action dispatched');
        assert.deepEqual(spy.firstCall.args[0], showMenu(MENU_STATE_NONE), 'is correct action');
    });
});
