import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { SettingsBannerContainer, QUALITY_CHANGE_DURATION } from 'ui/containers/settings/settings-banner';
import { SettingsBanner } from 'ui/components/settings/settings-banner';
import { mockTranslateFunc } from 'tests/utils/translate-hoc-helpers';

const DEFAULT_ARGS = Object.freeze({
    availableQualities: [],
    currentQuality: '',
    playbackRate: 1.0,
    t() {},
    windowObj: {},
});

function renderSettingsBanner(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <SettingsBannerContainer {...args} />;
    return shallow(component);
}

reactTest('ui | containers | settings | settings-banner', function() {
    QUnit.test('if hasChangedQuality and canRenderBanner, render a banner with the current quality', function(assert) {
        const fakeQualityGroup = 'low1';
        const fakeQualityName = 'wow is this even watchable';
        const availableQualities = [{
            group: fakeQualityGroup,
            name: fakeQualityName,
        }];
        const container = renderSettingsBanner({
            availableQualities,
            currentQuality: fakeQualityGroup,
            t: mockTranslateFunc,
        });
        container.setState({
            canRenderBanner: true,
            hasChangedQuality: true,
        });

        assert.equal(container.find(SettingsBanner).props().bannerLabel, fakeQualityName, 'show the quality label');
        assert.equal(container.find(SettingsBanner).props().showBanner, true, 'display the banner');
    });

    QUnit.test('if props change quality, set new state to show banner and timeout to remove banner', function(assert) {
        const timeoutId = QUnit.config.current.testId;
        const fakeWindow = {
            setTimeout: sinon.stub().returns(timeoutId),
        };
        const quality1 = 'low';
        const quality2 = 'medium';
        const container = renderSettingsBanner({
            currentQuality: quality1,
            windowObj: fakeWindow,
        });

        container.setProps({
            currentQuality: quality2,
        });

        assert.equal(fakeWindow.setTimeout.callCount, 1, 'set timeout called once');
        assert.equal(fakeWindow.setTimeout.firstCall.args[1], QUALITY_CHANGE_DURATION, 'duration is as expected');
        assert.equal(container.state().hasChangedQuality, true, 'hasChangedQuality is true');
        assert.equal(container.instance().bannerHideTimeout, timeoutId, 'timeout id is saved in state');

        fakeWindow.setTimeout.firstCall.args[0]();

        assert.equal(container.state().hasChangedQuality, false, 'hasChangedQuality is false');
        assert.equal(container.instance().bannerHideTimeout, 0, 'timeout id is cleared from state');
    });

    QUnit.test('if props do not change quality, do not show banner', function(assert) {
        const fakeWindow = {
            setTimeout: sinon.spy(),
        };
        const availableQualities1 = [{
            group: 'low',
            name: 'so low',
        }];
        const availableQualities2 = [{
            group: 'medium',
            name: 'kinda decent, I guess',
        }];

        const container = renderSettingsBanner({
            availableQualities: availableQualities1,
            windowObj: fakeWindow,
        });

        container.setProps({
            availableQualities: availableQualities2,
        });

        assert.equal(fakeWindow.setTimeout.callCount, 0, 'set timeout not called');
        assert.equal(container.state().hasChangedQuality, false, 'not showing banner');
    });

    QUnit.test('when unmounting, clear timeout if it exists', function(assert) {
        const timeoutId = '12345';
        const fakeWindow = {
            setTimeout() {},
            clearTimeout: sinon.spy(),
        };

        const container = renderSettingsBanner({
            windowObj: fakeWindow,
        });

        container.instance().bannerHideTimeout = timeoutId;
        container.unmount();
        assert.equal(fakeWindow.clearTimeout.callCount, 1, 'clear timeout invoked');
        assert.equal(fakeWindow.clearTimeout.firstCall.args[0], timeoutId, 'correct timeout cleared');
    });

    QUnit.test('show the banner with the playback rate if it is not 1.0 and canRenderBanner', function(assert) {
        const playbackRate = 2.0;
        const container = renderSettingsBanner({
            playbackRate,
        });

        container.setState({
            canRenderBanner: true,
        });

        assert.equal(container.find(SettingsBanner).props().showBanner, true, 'show banner');
        assert.equal(container.find(SettingsBanner).props().bannerLabel, `${playbackRate}x`, 'has correct label');
    });

    QUnit.test('if quality has not changed, playback rate is 1, and canRenderBanner, label is empty', function(assert) {
        const container = renderSettingsBanner({
            playbackRate: 1.0,
            qualityName: 'low',
        });

        container.setState({
            canRenderBanner: true,
        });

        assert.equal(container.state().hasChangedQuality, false, 'quality has not changed');
        assert.equal(container.find(SettingsBanner).props().bannerLabel, '', 'label is empty');
    });

    QUnit.module('canRenderBanner', function() {
        QUnit.test('is set to true if quality changes or playbackrate is not 1.0', function(assert) {
            const fakeWindow = {
                setTimeout() {},
                clearTimeout() {},
            };

            const container = renderSettingsBanner({
                windowObj: fakeWindow,
            });

            assert.equal(container.state().canRenderBanner, false, 'is initialized to false');

            container.setProps({
                currentQuality: 'low',
            });

            container.update();

            assert.equal(container.state().canRenderBanner, true, 'is set to true after quality change');

            container.setState({
                canRenderBanner: false,
            });

            container.update();

            container.setProps({
                playbackRate: 2.0,
            });

            container.update();

            assert.equal(container.state().canRenderBanner, true, 'is set to true on non 1.0 playback rate');
        });

        QUnit.test('determines if settings banner component is rendered', function(assert) {
            const fakeWindow = {
                setTimeout() {},
                clearTimeout() {},
            };

            const container = renderSettingsBanner({
                windowObj: fakeWindow,
            });

            assert.equal(container.type(), null, 'banner component is not rendered');

            container.setState({
                canRenderBanner: true,
            });

            assert.ok(container.find(SettingsBanner), 'banner component is rendered');
        });
    });
});
