import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { StatsContainer, mapStateToProps } from 'ui/containers/stats/stats';
import { Stats as StatsComponent } from 'ui/components/stats/stats';
import { CONTENT_MODE_LIVE } from 'stream/twitch-live';

const DEFAULT_STATS = Object.freeze({
    bufferSize: 0,
    displayResolution: '',
    skippedFrames: 0,
    fps: 0,
    hlsLatencyBroadcaster: 0,
    hlsLatencyEncoder: 0,
    memoryUsage: 0,
    playbackRate: 0,
    videoResolution: '',
    backendVersion: '',
});

const DEFAULT_ARGS = Object.freeze({
    isLive: true,
    stats: DEFAULT_STATS,
    displayed: false,
    closeStats() {},
});

function renderStats(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <StatsContainer {...args} />;
    return shallow(component);
}

reactTest('ui | containers | stats | stats', function() {
    QUnit.test('renders null if displayed is false', function(assert) {
        const component = renderStats();
        assert.equal(component.type(), null);
    });

    QUnit.test('renders a StatsComponent if displayed is true', function(assert) {
        const component = renderStats({
            displayed: true,
        });

        assert.equal(component.type(), StatsComponent);
    });

    QUnit.test('passes closeStats to child component', function(assert) {
        const closeStats = sinon.spy();
        const component = renderStats({
            displayed: true,
            closeStats,
        });

        component.props().onClose();
        assert.equal(closeStats.callCount, 1);
    });

    QUnit.test('passes props to child component', function(assert) {
        const component = renderStats({
            displayed: true,
        });
        const expectedProps = {
            isLive: DEFAULT_ARGS.isLive,
            stats: DEFAULT_ARGS.stats,
            onClose: component.instance().onClose,
        };

        assert.deepEqual(expectedProps, component.props());
    });

    QUnit.test('mapStateToProps properly populates props', function(assert) {
        const videoStats = assign({}, DEFAULT_STATS);
        const fakeStoreState = {
            stream: {
                contentType: CONTENT_MODE_LIVE,
            },
            stats: {
                videoStats,
                displayed: true,
            },
        };
        const resultProps = mapStateToProps(fakeStoreState);
        const expectedProps = {
            isLive: true,
            stats: videoStats,
            displayed: true,
        };

        assert.deepEqual(expectedProps, resultProps);
    });
});
