import React from 'react';
import assign from 'lodash/assign';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { init as initStore } from 'state';
import { setExperiments } from 'actions/experiments';
import { StreamStatusComponent, mapStateToProps } from 'ui/containers/top-bar/stream-status';
import { IconComponent } from 'ui/components/buttons/base-icon';
import { CONTENT_MODE_LIVE } from 'stream/twitch-live';
import { CONTENT_MODE_VOD } from 'stream/twitch-vod';
import { TWILIGHT_PREMIERE_UPLOAD_FLOW } from 'experiments';
import { VOD_RECOMMENDATION_SCREEN } from 'actions/screen';
import { ONLINE_STATUS } from 'state/online-status';
import classNames from 'classnames';

const DEFAULT_ARGS = Object.freeze({
    streamType: '',
    i18n: {
        translate: input => input,
    },
    isOnline: false,
    inCollection: false,
    showingRecommendations: false,
});

const STREAMSTATUS_CLASSES = classNames(
    'player-streamstatus'
);

function renderStreamStatusComponent(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <StreamStatusComponent {...args} />;
    return shallow(component);
}

reactTest('ui | containers | top-bar | stream-status', function() {
    QUnit.test('correctly maps state to props', function(assert) {
        const state = {
            lang: {},
            screen: [VOD_RECOMMENDATION_SCREEN],
            onlineStatus: ONLINE_STATUS,
            collection: {
                id: 1,
            },
            streamMetadata: {
                streamType: 'watch_party',
            },
            stream: {
                contentType: 'a stream type',
            },
        };

        assert.deepEqual(mapStateToProps(state), {
            streamType: state.stream.contentType,
            i18n: state.lang,
            isOnline: state.onlineStatus === ONLINE_STATUS,
            inCollection: Boolean(state.collection.id),
            isWatchParty: state.streamMetadata.streamType === 'watch_party',
            isRerun: state.streamMetadata.streamType === 'rerun',
            isPremiere: state.streamMetadata.streamType === 'premiere',
            showingRecommendations: state.screen[0] === VOD_RECOMMENDATION_SCREEN,
        });
    });

    QUnit.test('given default input, should create empty div', function(assert) {
        const streamStatus = renderStreamStatusComponent();
        assert.equal(streamStatus.type(), 'div', 'should create a div');
        assert.ok(streamStatus.hasClass(STREAMSTATUS_CLASSES), 'should create a div with expected classes');
        assert.equal(streamStatus.children().length, 0, 'should have no children');
    });

    QUnit.test('given online input, should create live icon and label', function(assert) {
        const input = {
            streamType: CONTENT_MODE_LIVE,
            isOnline: true,
        };
        const streamStatus = renderStreamStatusComponent(input);
        const liveStatusClasses = classNames(
            'player-streamstatus__icon',
            'player-streamstatus__icon--live',
            'qa-live-label'
        );

        assert.equal(streamStatus.type(), 'div', 'should create a div');
        assert.ok(streamStatus.hasClass(STREAMSTATUS_CLASSES), 'should create a div with expected classes');
        assert.equal(streamStatus.children().length, 1, 'should have 1 child');
        assert.ok(
            streamStatus.containsMatchingElement(
                <span>
                    <IconComponent svgClass={liveStatusClasses} iconId={'#icon_live'} />
                    <span className="player-streamstatus__label">Live</span>
                </span>
            ),
            'should contain expected span component'
        );
    });

    QUnit.test('given watch party input, should create vodcast icon and label', function(assert) {
        const input = {
            streamType: CONTENT_MODE_LIVE,
            isWatchParty: true,
            isOnline: true,
        };
        const streamStatus = renderStreamStatusComponent(input);
        const watchPartyStatusClasses = classNames(
            'player-streamstatus__icon',
            'player-streamstatus__icon--watchparty'
        );

        assert.equal(streamStatus.type(), 'div', 'should create a div');
        assert.ok(streamStatus.hasClass(STREAMSTATUS_CLASSES), 'should create a div with expected classes');
        assert.equal(streamStatus.children().length, 1, 'should have 1 child');
        assert.ok(
            streamStatus.containsMatchingElement(
                <span>
                    <IconComponent svgClass={watchPartyStatusClasses} iconId={'#icon_vodcast'} />
                    <span className="player-streamstatus__label">Vodcast</span>
                </span>
            ),
            'should contain expected span component'
        );
    });

    QUnit.test('given a premiere when experiment enabled, should create premiere icon and label', function(assert) {
        const input = {
            streamType: CONTENT_MODE_LIVE,
            isPremiere: true,
            isOnline: true,
        };

        this.state = initStore();

        const createExperiment = value => {
            return {
                get: uuid => {
                    switch (uuid) {
                    case TWILIGHT_PREMIERE_UPLOAD_FLOW:
                        return Promise.resolve(value);
                    }
                },
            };
        };

        this.state.dispatch(setExperiments(createExperiment('yes')));

        const streamStatus = renderStreamStatusComponent(input);
        const premiereStatusClasses = classNames(
            'player-streamstatus__icon',
            'player-streamstatus__icon--premiere'
        );

        assert.equal(streamStatus.type(), 'div', 'should create a div');
        assert.ok(streamStatus.hasClass(STREAMSTATUS_CLASSES), 'should create a div with expected classes');
        assert.equal(streamStatus.children().length, 1, 'should have 1 child');
        assert.ok(
            streamStatus.containsMatchingElement(
                <span>
                    <IconComponent svgClass={premiereStatusClasses} iconId={'#icon_premiere'} />
                    <span className="player-streamstatus__label">Premiere</span>
                </span>
            ),
            'should contain expected span component'
        );
    });

    QUnit.test('given a premiere with experiment disabled, creates vodcast icon and label', function(assert) {
        const input = {
            streamType: CONTENT_MODE_LIVE,
            isPremiere: true,
            isOnline: true,
        };

        this.state = initStore();

        const createExperiment = value => {
            return {
                get: uuid => {
                    switch (uuid) {
                    case TWILIGHT_PREMIERE_UPLOAD_FLOW:
                        return Promise.resolve(value);
                    }
                },
            };
        };

        this.state.dispatch(setExperiments(createExperiment('no')));

        const streamStatus = renderStreamStatusComponent(input);
        const watchPartyStatusClasses = classNames(
            'player-streamstatus__icon',
            'player-streamstatus__icon--watchparty'
        );
        assert.equal(streamStatus.type(), 'div', 'should create a div');
        assert.ok(streamStatus.hasClass(STREAMSTATUS_CLASSES), 'should create a div with expected classes');
        assert.equal(streamStatus.children().length, 1, 'should have 1 child');
        assert.ok(
            streamStatus.containsMatchingElement(
                <span>
                    <IconComponent svgClass={watchPartyStatusClasses} iconId={'#icon_vodcast'} />
                    <span className="player-streamstatus__label">Vodcast</span>
                </span>
            ),
            'should contain expected span component'
        );
    });

    QUnit.test('given a rerun when experiment enabled, should create rerun icon and label', function(assert) {
        const input = {
            streamType: CONTENT_MODE_LIVE,
            isRerun: true,
            isOnline: true,
        };

        this.state = initStore();

        const createExperiment = value => {
            return {
                get: uuid => {
                    switch (uuid) {
                    case TWILIGHT_PREMIERE_UPLOAD_FLOW:
                        return Promise.resolve(value);
                    }
                },
            };
        };

        this.state.dispatch(setExperiments(createExperiment('yes')));

        const streamStatus = renderStreamStatusComponent(input);
        const rerunStatusClasses = classNames(
            'player-streamstatus__icon',
            'player-streamstatus__icon--rerun'
        );

        assert.equal(streamStatus.type(), 'div', 'should create a div');
        assert.ok(streamStatus.hasClass(STREAMSTATUS_CLASSES), 'should create a div with expected classes');
        assert.equal(streamStatus.children().length, 1, 'should have 1 child');
        assert.ok(
            streamStatus.containsMatchingElement(
                <span>
                    <IconComponent svgClass={rerunStatusClasses} iconId={'#icon_rerun'} />
                    <span className="player-streamstatus__label">Rerun</span>
                </span>
            ),
            'should contain expected span component'
        );
    });

    QUnit.test('given a rerun with experiment disabled, creates vodcast icon and label', function(assert) {
        const input = {
            streamType: CONTENT_MODE_LIVE,
            isRerun: true,
            isOnline: true,
        };

        this.state = initStore();

        const createExperiment = value => {
            return {
                get: uuid => {
                    switch (uuid) {
                    case TWILIGHT_PREMIERE_UPLOAD_FLOW:
                        return Promise.resolve(value);
                    }
                },
            };
        };

        this.state.dispatch(setExperiments(createExperiment('no')));

        const streamStatus = renderStreamStatusComponent(input);
        const watchPartyStatusClasses = classNames(
            'player-streamstatus__icon',
            'player-streamstatus__icon--watchparty'
        );
        assert.equal(streamStatus.type(), 'div', 'should create a div');
        assert.ok(streamStatus.hasClass(STREAMSTATUS_CLASSES), 'should create a div with expected classes');
        assert.equal(streamStatus.children().length, 1, 'should have 1 child');
        assert.ok(
            streamStatus.containsMatchingElement(
                <span>
                    <IconComponent svgClass={watchPartyStatusClasses} iconId={'#icon_vodcast'} />
                    <span className="player-streamstatus__label">Vodcast</span>
                </span>
            ),
            'should contain expected span component'
        );
    });

    QUnit.test('given offline input, should create offline label', function(assert) {
        const input = {
            streamType: CONTENT_MODE_LIVE,
        };
        const streamStatus = renderStreamStatusComponent(input);
        const offlineStatusClasses = classNames(
            'player-streamstatus__icon',
            'player-streamstatus__icon--offline'
        );
        assert.equal(streamStatus.type(), 'div', 'should create a div');
        assert.ok(streamStatus.hasClass(STREAMSTATUS_CLASSES), 'should create a div with expected classes');
        assert.equal(streamStatus.children().length, 1, 'should have 1 child');
        assert.ok(
            streamStatus.containsMatchingElement(
                <span>
                    <IconComponent svgClass={offlineStatusClasses} iconId={'#icon_offline'} />
                    <span className="player-streamstatus__label">Offline</span>
                </span>
            ),
            'should contain expected span component'
        );
    });

    QUnit.test('if in vod, should not generate label', function(assert) {
        const input = {
            streamType: CONTENT_MODE_VOD,
        };
        const streamStatus = renderStreamStatusComponent(input);

        assert.equal(streamStatus.type(), 'div', 'should create a div');
        assert.ok(streamStatus.hasClass(STREAMSTATUS_CLASSES), 'should create a div with expected classes');
        assert.equal(streamStatus.children().length, 0, 'should have 0 child');
    });

    QUnit.test('given collection, should not generate label', function(assert) {
        const input = {
            inCollection: true,
        };
        const streamStatus = renderStreamStatusComponent(input);

        assert.equal(streamStatus.type(), 'div', 'should create a div');
        assert.ok(streamStatus.hasClass(STREAMSTATUS_CLASSES), 'should create a div with expected classes');
        assert.equal(streamStatus.children().length, 0, 'should have 0 child');
    });

    QUnit.test('if show offline recs, should not generate label', function(assert) {
        const input = {
            showingRecommendations: true,
        };
        const streamStatus = renderStreamStatusComponent(input);

        assert.equal(streamStatus.type(), 'div', 'should create a div');
        assert.ok(streamStatus.hasClass(STREAMSTATUS_CLASSES), 'should create a div with expected classes');
        assert.equal(streamStatus.children().length, 0, 'should have 0 child');
    });
});
