import React from 'react';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { TwitchEverywherePanel, AUTHENTICATION_REQUIRED, SUBSCRIBE,
         TE_NAMESPACE } from 'ui/containers/twitch-everywhere-panel';
import { TwitchEverywhereFollowPanel } from 'ui/components/follow-panels/twitch-everywhere-panel';
import { PLAYER_TWITCH_EVERYWHERE } from 'util/player-type';
import { NOTIFICATIONS_BUTTON_VIEW } from 'actions/follow';

const CHANNEL_ID = 1;
const USER_ID = 2;
const TARGET_ORIGIN = 'http://example.com';

reactTest('ui | containers | twitch-everywhere-panel', function(hooks) {
    hooks.beforeEach(function() {
        this.args = {
            avatarURL: 'avatar-url',
            channelName: 'channel-name',
            channelId: CHANNEL_ID,
            userId: USER_ID,
            isLoggedIn: true,
            isFollowing: false,
            didFetchFollow: true,
            didFetchSubscriptions: true,
            followChannel: sinon.spy(),
            unfollowChannel: sinon.spy(),
            window: {
                parent: {
                    postMessage: sinon.spy(),
                },
            },
            notificationsEnabled: false,
            analytics: { trackEvent: sinon.spy() },
            targetOrigin: TARGET_ORIGIN,
        };
    });

    QUnit.test('when data is fetched, renders the TwitchEverywhereFollowPanel', function(assert) {
        const component = shallow(
            <TwitchEverywherePanel {...this.args} />
        );

        const panelProps = component.find(TwitchEverywhereFollowPanel).props();

        assert.equal(panelProps.avatarURL, this.args.avatarURL, 'passes avatarURL');

        panelProps.onFollowChannel();
        assert.ok(
            this.args.followChannel.calledWith(this.args.userId, this.args.channelId),
            'creates followChannel callback bound to userId and channelId'
        );

        panelProps.onUnfollowChannel();
        assert.ok(
            this.args.unfollowChannel.calledWith(this.args.userId, this.args.channelId),
            'creates unfollowChannel callback bound to userId and channelId'
        );

        panelProps.onSubscribe();
        assert.ok(
            this.args.window.parent.postMessage.calledWith({
                eventName: SUBSCRIBE,
                namespace: TE_NAMESPACE,
                params: {
                    channelName: this.args.channelName,
                },
            }, TARGET_ORIGIN),
            'sends subscription event to the TwitchEverywhere window'
        );
    });

    QUnit.test('when didFetchFollow is false, renders null', function(assert) {
        const component = shallow(
            <TwitchEverywherePanel {...this.args} didFetchFollow={false} />
        );

        assert.equal(component.type(), null);
    });

    QUnit.test('when didFetchSubscriptions is false, renders null', function(assert) {
        const component = shallow(
            <TwitchEverywherePanel {...this.args} didFetchSubscriptions={false} />
        );

        assert.equal(component.type(), null);
    });

    QUnit.test('when viewing user\'s own channel, renders null', function(assert) {
        const component = shallow(
            <TwitchEverywherePanel {...this.args} channelId={1} userId={1} />
        );

        assert.equal(component.type(), null);
    });

    QUnit.test('when user not logged in, followChannel sends an authentication event', function(assert) {
        const component = shallow(
            <TwitchEverywherePanel
                {...this.args}
                isLoggedIn={false}
                didFetchFollow={false}
                didFetchSubscriptions={false}
            />
        );

        const panelProps = component.find(TwitchEverywhereFollowPanel).props();
        panelProps.onFollowChannel();

        assert.equal(this.args.followChannel.callCount, 0, 'does not call followChannel');

        assert.ok(
            this.args.window.parent.postMessage.calledWith({
                eventName: AUTHENTICATION_REQUIRED,
                namespace: TE_NAMESPACE,
                params: {},
            }, TARGET_ORIGIN),
            'sends authentication event to the TwitchEverywhere window'
        );
    });

    QUnit.test('passes a callback to track viewing', function(assert) {
        const component = shallow(<TwitchEverywherePanel {...this.args} />);
        component.instance().componentDidMount();

        assert.ok(
            this.args.analytics.trackEvent.calledWith(NOTIFICATIONS_BUTTON_VIEW, {
                /* eslint-disable camelcase */
                follow_state: false,
                notif_state: false,
                /* eslint-enable */
                src: PLAYER_TWITCH_EVERYWHERE,
            }),
            'called with follow_state, notif_state, and src'
        );
    });
});
