import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import { VolumeSliderComponent, sliderClassNames,
         mapDispatchToProps, mapStateToProps } from 'ui/containers/volume-slider';
import { Slider } from 'ui/components/slider';
import * as TrackingActions from 'actions/analytics-tracker';
import * as VideoApiActions from 'actions/video-api';
import { mockTranslateFunc } from 'tests/utils/translate-hoc-helpers';

const DEFAULT_ARGS = Object.freeze({
    t: mockTranslateFunc,
    muted: false,
    mutePlayer() {},
    changeVolume() {},
    volume: 100,
    trackVolumeButtonClick: () => {},
    isPlayingSound: true,
    showVolumeSlider: true,
});

function renderVolumeSlider(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <VolumeSliderComponent {...args} />;
    return shallow(component);
}

const FAKE_TRACK_EVENT_ACTION = 'a fake track event action';
const FAKE_CHANGE_VOLUME_ACTION = 'a fake change volume action';
const FAKE_MUTE_PLAYER_ACTION = 'a fake mute player action';

reactTest('ui | containers | volume-slider', function(hooks) {
    hooks.beforeEach(function() {
        sinon.stub(TrackingActions, 'trackEvent', (name, payload) => ({
            name,
            payload,
            type: FAKE_TRACK_EVENT_ACTION,
        }));
        sinon.stub(VideoApiActions, 'mutePlayer', (muted, automated) => ({
            muted,
            automated,
            type: FAKE_MUTE_PLAYER_ACTION,
        }));
        sinon.stub(VideoApiActions, 'changeVolume', volume => ({
            volume,
            type: FAKE_CHANGE_VOLUME_ACTION,
        }));
    });

    hooks.afterEach(function() {
        TrackingActions.trackEvent.restore();
        VideoApiActions.mutePlayer.restore();
        VideoApiActions.changeVolume.restore();
    });

    QUnit.test('correctly maps state to props', function(assert) {
        const state = {
            ui: {
                showVolumeSlider: false,
            },
            playback: {
                muted: false,
                volume: 0.1,
            },
        };

        assert.deepEqual(mapStateToProps(state), {
            showVolumeSlider: state.ui.showVolumeSlider,
            muted: state.playback.muted,
            volume: state.playback.volume,
            isPlayingSound: !state.playback.muted && state.playback.volume !== 0,
        });
    });

    QUnit.module('correctly maps dispatches to props', function() {
        QUnit.test('trackVolumeButtonClick', function(assert) {
            const dispatchSpy = sinon.spy();
            const dispatchProps = mapDispatchToProps(dispatchSpy);

            dispatchProps.trackVolumeButtonClick(false);
            assert.equal(dispatchSpy.firstCall.args[0].type, FAKE_TRACK_EVENT_ACTION, 'correct action type');
            assert.equal(dispatchSpy.firstCall.args[0].name, 'player_click', 'correct event name');
            assert.deepEqual(
                dispatchSpy.firstCall.args[0].payload,
                {
                    component: 'volume_button',
                    // eslint-disable-next-line camelcase
                    component_state: 'unmute',
                },
                'correct event payload'
            );

            dispatchSpy.reset();

            dispatchProps.trackVolumeButtonClick(true);
            assert.equal(dispatchSpy.firstCall.args[0].type, FAKE_TRACK_EVENT_ACTION, 'correct action type');
            assert.equal(dispatchSpy.firstCall.args[0].name, 'player_click', 'correct event name');
            assert.deepEqual(
                dispatchSpy.firstCall.args[0].payload,
                {
                    component: 'volume_button',
                    // eslint-disable-next-line camelcase
                    component_state: 'mute',
                },
                'correct event payload'
            );
        });

        QUnit.test('mutePlayer', function(assert) {
            const dispatchSpy = sinon.spy();
            const dispatchProps = mapDispatchToProps(dispatchSpy);

            dispatchProps.mutePlayer(false);
            assert.equal(dispatchSpy.firstCall.args[0].type, FAKE_MUTE_PLAYER_ACTION, 'correct action type');
            assert.equal(dispatchSpy.firstCall.args[0].muted, false, 'passes in muted value');
            assert.equal(dispatchSpy.firstCall.args[0].automated, false, 'set automated value to false');
        });

        QUnit.test('changeVolume', function(assert) {
            const dispatchSpy = sinon.spy();
            const dispatchProps = mapDispatchToProps(dispatchSpy);
            const volume = 0.5;

            dispatchProps.changeVolume(volume);
            assert.equal(dispatchSpy.firstCall.args[0].type, FAKE_CHANGE_VOLUME_ACTION, 'correct action type');
            assert.equal(dispatchSpy.firstCall.args[0].volume, volume, 'passes in volume');
        });
    });

    QUnit.test('returns a div', function(assert) {
        const component = renderVolumeSlider();
        assert.equal(component.type(), 'div');
    });

    QUnit.test('if showVolumeSlider is false, render null', function(assert) {
        const component = renderVolumeSlider({
            showVolumeSlider: false,
        });
        assert.equal(component.type(), null);
    });

    QUnit.test('if isPlayingSound is false, unmute icon is displayed', function(assert) {
        const component = renderVolumeSlider({
            isPlayingSound: false,
        });
        assert.equal(1, component.find('.unmute-button').length);
        assert.equal(0, component.find('.mute-button').length);
    });

    QUnit.test('if isPlayingSound is true, mute icon is displayed', function(assert) {
        const component = renderVolumeSlider({
            isPlayingSound: true,
        });

        assert.equal(0, component.find('.unmute-button').length);
        assert.equal(1, component.find('.mute-button').length);
    });

    QUnit.test('if not muted, slider value is volume value', function(assert) {
        const component = renderVolumeSlider({
            muted: false,
            volume: 0.5,
        });

        assert.ok(component.containsMatchingElement(
            <Slider
                max={1}
                min={0}
                value={0.5}
                classNames={sliderClassNames}
            />
        ));
    });

    QUnit.test('if muted, slider value is zero', function(assert) {
        const component = renderVolumeSlider({
            muted: true,
        });
        assert.ok(component.containsMatchingElement(
            <Slider
                max={1}
                min={0}
                value={0}
                classNames={sliderClassNames}
            />
        ));
    });

    // eslint-disable-next-line max-len
    QUnit.test('on icon click, if isPlayingSound is false, and volume is not 0, unmute Player, and track unmute click', function(assert) {
        const mutePlayerSpy = sinon.spy();
        const trackButtonClickSpy = sinon.spy();
        const component = renderVolumeSlider({
            volume: 0.1,
            isPlayingSound: false,
            mutePlayer: mutePlayerSpy,
            trackVolumeButtonClick: trackButtonClickSpy,
        });

        mutePlayerSpy.reset();
        trackButtonClickSpy.reset();

        component.find('button').simulate('click');
        assert.equal(mutePlayerSpy.firstCall.args[0], false, 'mutePlayer called with false to unmute player');
        assert.equal(trackButtonClickSpy.callCount, 1, 'trackVolumeButtonClick called');
        assert.equal(trackButtonClickSpy.firstCall.args[0], false, 'trackVolumeButtonClick called with false');
    });

    // eslint-disable-next-line max-len
    QUnit.test('on icon click, if isPlayingSound is false and volume is 0, unmute Player, change volume to 0.1, track unmute click', function(assert) {
        const mutePlayerSpy = sinon.spy();
        const changeVolumeSpy = sinon.spy();
        const trackButtonClickSpy = sinon.spy();
        const component = renderVolumeSlider({
            volume: 0,
            isPlayingSound: false,
            mutePlayer: mutePlayerSpy,
            trackVolumeButtonClick: trackButtonClickSpy,
            changeVolume: changeVolumeSpy,
        });

        mutePlayerSpy.reset();
        changeVolumeSpy.reset();
        trackButtonClickSpy.reset();

        component.find('button').simulate('click');
        assert.equal(mutePlayerSpy.firstCall.args[0], false, 'mutePlayer called with false to unmute player');
        assert.equal(changeVolumeSpy.firstCall.args[0], 0.1, 'changeVolume called with 0.1');
        assert.equal(trackButtonClickSpy.callCount, 1, 'trackVolumeButtonClick called');
        assert.equal(trackButtonClickSpy.firstCall.args[0], false, 'trackVolumeButtonClick called with false');
    });

    QUnit.test('on icon click, if isPlayingSound is true, mute Player, track mute click', function(assert) {
        const mutePlayerSpy = sinon.spy();
        const trackButtonClickSpy = sinon.spy();
        const component = renderVolumeSlider({
            volume: 0.5,
            isPlayingSound: true,
            mutePlayer: mutePlayerSpy,
            trackVolumeButtonClick: trackButtonClickSpy,
        });

        mutePlayerSpy.reset();
        trackButtonClickSpy.reset();

        component.find('button').simulate('click');
        assert.equal(mutePlayerSpy.firstCall.args[0], true, 'mutePlayer called with true to mute player');
        assert.equal(trackButtonClickSpy.callCount, 1, 'trackVolumeButtonClick called');
        assert.equal(trackButtonClickSpy.firstCall.args[0], true, 'trackVolumeButtonClick called with true');
    });

    QUnit.test('on slider drag, invokes changeVolume and unmutes player', function(assert) {
        const changeVolumeSpy = sinon.spy();
        const mutePlayerSpy = sinon.spy();
        const component = renderVolumeSlider({
            mutePlayer: mutePlayerSpy,
            changeVolume: changeVolumeSpy,
        });

        changeVolumeSpy.reset();
        mutePlayerSpy.reset();

        component.find(Slider).props().dragHandlers.onDrag(0.5);

        assert.equal(changeVolumeSpy.firstCall.args[0], 0.5, 'changeVolume called with value');
        assert.equal(mutePlayerSpy.firstCall.args[0], false, 'mutePlayer called with false to unmute player');
    });

    QUnit.test('on slider stop, invokes changeVolume and unmutes player', function(assert) {
        const changeVolumeSpy = sinon.spy();
        const mutePlayerSpy = sinon.spy();
        const component = renderVolumeSlider({
            mutePlayer: mutePlayerSpy,
            changeVolume: changeVolumeSpy,
        });

        changeVolumeSpy.reset();
        mutePlayerSpy.reset();

        component.find(Slider).props().dragHandlers.onStop(0.5);

        assert.equal(changeVolumeSpy.firstCall.args[0], 0.5, 'changeVolume called with value');
        assert.equal(mutePlayerSpy.firstCall.args[0], false, 'mutePlayer called with false to unmute player');
    });

    QUnit.test('on slider click, invokes changeVolume and unmutes player', function(assert) {
        const changeVolumeSpy = sinon.spy();
        const mutePlayerSpy = sinon.spy();
        const component = renderVolumeSlider({
            mutePlayer: mutePlayerSpy,
            changeVolume: changeVolumeSpy,
        });

        changeVolumeSpy.reset();
        mutePlayerSpy.reset();

        component.find(Slider).props().onClick(0.5);

        assert.equal(changeVolumeSpy.firstCall.args[0], 0.5, 'changeVolume called with value');
        assert.equal(mutePlayerSpy.firstCall.args[0], false, 'mutePlayer called with false to unmute player');
    });
});
