import React from 'react';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import sinon from 'sinon';

import { init as initStore } from 'state';
import { PlayerUIControls } from 'ui/controls';
import { PlayerUIInfo } from 'ui/info';
import { LoadingOverlay } from 'ui/containers/overlays/loading-overlay';
import { BottomControls } from 'ui/containers/bottom-controls';
import { ClipsMinutesWatched } from 'ui/player-types/clips/utils/tracking/clips-minutes-watched';
import { ClipsBenchmarkTracking } from 'ui/player-types/clips/utils/tracking/clips-benchmark-tracking';
import { ClipsLoadingTimeSpentTracker } from 'ui/player-types/clips/utils/tracking/clips-loading-time-spent-tracker';
import ClipsEmbedPlayer from 'ui/player-types/clips/embed';
import assign from 'lodash/assign';

const uiModules = Object.freeze([
    PlayerUIInfo,
    PlayerUIControls,
    ClipsMinutesWatched,
    ClipsBenchmarkTracking,
    ClipsLoadingTimeSpentTracker,
]);

const DEFAULT_PLAYER = Object.freeze({
    getCurrentTime: () => 0,
    addEventListener: () => {},
    removeEventListener: () => {},
});

function renderClipsEmbedPlayer() {
    const store = initStore();
    const state = store.getState();
    store.getState = () => assign({}, state, {
        analyticsTracker: {
            clipsTrackEvent: () => {},
        },
    });
    const root = window.document.createElement('div');

    // TODO: Remove once ui/controls is removed in VP-2645
    const $fullscreenOverlay = window.document.createElement('div');
    $fullscreenOverlay.classList.add('js-control-fullscreen-overlay');
    root.appendChild($fullscreenOverlay);

    window.document.getElementById('qunit-fixture').appendChild(root);

    const component = <ClipsEmbedPlayer store={store} root={root} player={DEFAULT_PLAYER} />;
    return shallow(component, {
        context: {
            store,
        },
    });
}

reactTest('ui | player-types | clips | embed', function(hooks) {
    hooks.beforeEach(function() {
        this.component = renderClipsEmbedPlayer();
    });

    hooks.afterEach(function() {
        // To ensure that the legacy ui modules are destroyed
        this.component.unmount();
    });

    QUnit.test('renders a div', function(assert) {
        assert.equal(this.component.type(), 'div');
    });

    QUnit.test('has a LoadingOverlay', function(assert) {
        assert.equal(this.component.find(LoadingOverlay).length, 1);
    });

    QUnit.test('has BottomControls', function(assert) {
        assert.equal(this.component.find(BottomControls).length, 1);
    });

    QUnit.test('on mount, creates correct ui modules sets it in state', function(assert) {
        const legacyUIState = this.component.state().legacyUI;
        legacyUIState.forEach(createdModule => {
            const isValidModule = uiModules.reduce((acc, module) => {
                return acc || createdModule instanceof module;
            }, false);
            assert.equal(isValidModule, true, `${createdModule} is not an expected type`);
        });
    });

    QUnit.test('on unmount, destroys ui modules', function(assert) {
        const legacyUIState = this.component.state().legacyUI;
        legacyUIState.forEach(createdModule => {
            sinon.spy(createdModule, 'destroy');
        });
        this.component.unmount();
        legacyUIState.forEach(createdModule => {
            assert.ok(createdModule.destroy.called, `${createdModule}'s destroy should be called`);
        });
    });
});
