import React from 'react';
import assign from 'lodash/assign';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import {
    CardContentPreviewComponent,
} from 'ui/player-types/clips/overlays/clip-postplay-recommendations/card-content-preview';
import { mockTranslateFunc } from 'tests/utils/translate-hoc-helpers';
import {
    ITEM_TYPE_CLIP,
    ITEM_TYPE_LIVE,
    ITEM_TYPE_VOD,
} from 'ui/player-types/clips/utils/embed-postplay-overlay-selectors';
import { ClipTinySVG } from 'ui/player-types/clips/common/svg/clip-tiny';
import { VodTinySVG } from 'ui/player-types/clips/common/svg/vod-tiny';

const DEFAULT_ARGS = Object.freeze({
    linkUrl: 'linkUrl',
    itemType: ITEM_TYPE_CLIP,
    imageUrl: 'imageUrl',
    title: 'title',
    viewCount: 100,
    game: 'game',
    displayName: 'displayName',
    trackSelectionClick: () => {},
    t: mockTranslateFunc,
    cardDimensions: {
        thumb: {
            width: 30,
            height: 40,
        },
        card: {
            width: 50,
            height: 60,
        },
    },
});

function renderComponent(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <CardContentPreviewComponent {...args} />;
    return shallow(component);
}

reactTest('ui | player-types | clips | overlays | clip-postplay-recommendations | card-content-preview', function() {
    QUnit.test('should set dimensions for two divs', function(assert) {
        const component = renderComponent({ itemType: ITEM_TYPE_CLIP });

        assert.equal(component.find({
            style: {
                width: DEFAULT_ARGS.cardDimensions.thumb.width,
                height: DEFAULT_ARGS.cardDimensions.thumb.height,
            },
        }).length, 1);

        assert.equal(component.find({
            style: {
                width: DEFAULT_ARGS.cardDimensions.card.width,
                height: DEFAULT_ARGS.cardDimensions.card.height,
            },
        }).length, 1);
    });

    QUnit.test('shows the content for a CLIP', function(assert) {
        const component = renderComponent({ itemType: ITEM_TYPE_CLIP });

        // Clip
        assert.equal(component.find(ClipTinySVG).length, 1, 'ClipTinySVG visibility');
        assert.equal(component.text().includes('Clip'), true, 'includes Clip');
        assert.equal(component.text().includes('100 view • game'), true, '100 view • game');

        // Vod
        assert.equal(component.find(VodTinySVG).length, 0, 'VodTinySVG visibility');
        assert.equal(component.text().includes('Full Video'), false);
        assert.equal(component.text().includes('Continue Watching'), false);
        assert.equal(component.text().includes('displayName playing game'), false);

        // Live
        assert.equal(component.find('.cl-live-indicator').length, 0, 'live visible');
        assert.equal(component.text().includes('Live'), false, 'live pill visible');
    });

    QUnit.test('shows the content for a VOD', function(assert) {
        const component = renderComponent({ itemType: ITEM_TYPE_VOD });

        // Clip
        assert.equal(component.find(ClipTinySVG).length, 0, 'ClipTinySVG visibility');
        assert.equal(component.text().includes('Clip'), false, 'includes Clip');
        assert.equal(component.text().includes('100 view • game'), false, '100 view • game');

        // Vod
        assert.equal(component.find(VodTinySVG).length, 1, 'VodTinySVG visibility');
        assert.equal(component.text().includes('Full Video'), true);
        assert.equal(component.text().includes('Continue Watching'), true);
        assert.equal(component.text().includes('displayName playing game'), true);

        // Live
        assert.equal(component.find('.cl-live-indicator').length, 0, 'live visible');
        assert.equal(component.text().includes('Live'), false, 'live pill visible');
    });

    QUnit.test('shows the content for a LIVE', function(assert) {
        const component = renderComponent({ itemType: ITEM_TYPE_LIVE });

        // Clip
        assert.equal(component.find(ClipTinySVG).length, 0, 'ClipTinySVG visibility');
        assert.equal(component.text().includes('Clip'), false, 'includes Clip');
        assert.equal(component.text().includes('100 view • game'), false, '100 view • game');

        // Vod
        assert.equal(component.find(VodTinySVG).length, 0, 'VodTinySVG visibility');
        assert.equal(component.text().includes('Full Video'), false);
        assert.equal(component.text().includes('Continue Watching'), false);

        // Live
        assert.equal(component.find('.cl-live-indicator').length, 1, 'live visible');
        assert.equal(component.text().includes('Live'), true, 'live pill visible');
        assert.equal(component.text().includes('displayName playing game'), true);
    });
});
