import React from 'react';
import assign from 'lodash/assign';
import sinon from 'sinon';
import { reactTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import {
    ClipPostplayRecommendationsComponent,
} from 'ui/player-types/clips/overlays/clip-postplay-recommendations/clip-postplay-recommendations';
import {
    PostplayContentTwoLayout,
} from 'ui/player-types/clips/overlays/clip-postplay-recommendations/postplay-content-two-layout';
import {
    TRACKING_ITEM_SECTION_CLICK,
    TRACKING_ITEM_SECTION_LOAD,
} from 'ui/player-types/clips/utils/tracking/clips-tracking-constants';

const DEFAULT_ARGS = Object.freeze({
    channelName: 'name',
    showOverlay: true,
    postPlayContentData: [{
        itemType: 'itemType',
        linkUrl: 'linkUrl',
        title: 'title',
        itemId: 'itemId',
        viewCount: 'viewCount',
        channel: 'channel',
        displayName: 'displayName',
        game: 'game',
    }],
    fetchRecommendationsSets: () => {},
    hasFetchedRelatedClips: false,
    hasFetchedBroadcasterInfo: false,
    clipSlug: 'clipSlug',
    playerDimensions: {
        width: 30,
        height: 40,
    },
    analyticsTracker: {
        clipsTrackEvent() {},
    },
});

function renderComponent(overrides = {}) {
    const args = assign({}, DEFAULT_ARGS, overrides);
    const component = <ClipPostplayRecommendationsComponent {...args} />;
    return shallow(component);
}

reactTest('ui | player-types | clips | overlays | clip-postplay-recommendations', function() {
    QUnit.test('should not render postplay when showOverlay is false', function(assert) {
        const component = renderComponent({ showOverlay: false });

        assert.equal(component.find(PostplayContentTwoLayout).length, 0);
    });

    QUnit.test('should not render postplay when showOverlay is false', function(assert) {
        const component = renderComponent({ showOverlay: true });

        assert.equal(component.find(PostplayContentTwoLayout).length, 1);
    });

    QUnit.test('should call fetchRecommendationsSets when it has not yet and there is a clip slug', function(assert) {
        const component = renderComponent();
        const fetchRecommendationsSets = sinon.spy();

        assert.equal(fetchRecommendationsSets.callCount, 0, 'not called yet');
        component.attemptedRecommendationsFetched = false;
        component.instance().componentWillReceiveProps(assign({}, DEFAULT_ARGS, {
            fetchRecommendationsSets,
            clipSlug: 'slug',
            channelName: 'name',
        }));

        assert.equal(fetchRecommendationsSets.callCount, 1, 'called');
    });

    QUnit.test('should preload data when it has not yet done so with required data', function(assert) {
        const component = renderComponent();
        assert.equal(component.state().isImagePreloadingComplete, false);

        component.instance().componentWillReceiveProps(assign({}, DEFAULT_ARGS, {
            hasFetchedRelatedClips: true,
            hasFetchedBroadcasterInfo: true,
            clipSlug: 'slug',
        }));

        assert.equal(component.state().isImagePreloadingComplete, true);
    });

    QUnit.test('should track when show overlay is true and it has not tracked yet', function(assert) {
        const clipsTrackEventSpy = sinon.spy();
        const component = renderComponent({
            analyticsTracker: {
                clipsTrackEvent: clipsTrackEventSpy,
            },
        });
        component.instance().trackedForVideoEnd = false;

        assert.equal(clipsTrackEventSpy.callCount, 0, 'not called yet');

        component.instance().componentWillReceiveProps(assign({}, DEFAULT_ARGS, {
            showOverlay: true,
        }));

        assert.equal(clipsTrackEventSpy.callCount, 1, 'called now');
        assert.equal(clipsTrackEventSpy.getCall(0).args[0], TRACKING_ITEM_SECTION_LOAD);
    });

    QUnit.test('should track click event when the click event handler is called', function(assert) {
        const clipsTrackEventSpy = sinon.spy();
        const component = renderComponent({
            analyticsTracker: {
                clipsTrackEvent: clipsTrackEventSpy,
            },
        });
        component.instance().trackedForVideoEnd = false;

        assert.equal(clipsTrackEventSpy.callCount, 0, 'not called yet');

        component.instance().handleTrackForClickGenerator({}, 0)();

        assert.equal(clipsTrackEventSpy.callCount, 1, 'called now');
        assert.equal(clipsTrackEventSpy.getCall(0).args[0], TRACKING_ITEM_SECTION_CLICK);
    });
});
