import React from 'react';
import ReactDOM from 'react-dom';
import { reactUnitTest } from 'tests/utils/react-test';
import { shallow } from 'enzyme';
import sinon from 'sinon';
import assign from 'lodash/assign';
import * as UI from 'ui/main';
import { uiLoader } from 'ui/ui-loader';
import { PlayerTypeLoader } from 'ui/player-type-loader';
import { init as initStore } from 'state';

const DEFAULT_OPTS = Object.freeze({
    player: 'site',
    muted: false,
});

reactUnitTest('ui | ui-loader ', function(hooks) {
    hooks.beforeEach(function() {
        this.video = {};
        this.root = document.createElement('div');
        this.store = initStore();
        this.clipGenerator = {};
        this.playerUIMock = sinon.stub(UI, 'PlayerUI').returns({ destroy() {} });
        this.reactDOMMock = sinon.stub(ReactDOM, 'render');
    });

    hooks.afterEach(function() {
        this.playerUIMock.restore();
        this.reactDOMMock.restore();
    });

    function testPlayerType(playerType) {
        QUnit.test(`uiLoader should return a destructable - playerType: ${playerType}`, function(assert) {
            const options = assign({}, DEFAULT_OPTS, {
                player: playerType,
            });
            const loadedUI = uiLoader(
                this.video,
                this.root,
                this.store,
                this.clipGenerator,
                options
            );
            assert.ok(loadedUI.hasOwnProperty('destroy'), 'has destroy method');
        });
    }

    testPlayerType('amazon_live');
    testPlayerType('site');

    QUnit.test('loads ui/main by default', function(assert) {
        uiLoader(
            this.video,
            this.root,
            this.store,
            this.clipGenerator,
            DEFAULT_OPTS
        );

        assert.equal(this.playerUIMock.callCount, 1);
    });

    QUnit.test('renders PlayerTypeLoader if part of ASYNC_PLAYER_TYPES', function(assert) {
        uiLoader(
            this.video,
            this.root,
            this.store,
            this.clipGenerator,
            { player: 'amazon_live' }
        );

        const mountPoint = this.root.getElementsByClassName('player-ui')[0];
        const rootComponent = shallow(this.reactDOMMock.firstCall.args[0]);

        assert.equal(
            this.reactDOMMock.callCount,
            1,
            'Render called once'
        );
        assert.equal(
            this.reactDOMMock.firstCall.args[1],
            mountPoint,
            'Mounted to correct div'
        );
        assert.ok(
            rootComponent.contains(<PlayerTypeLoader player={this.video} root={this.root} />),
            'renders PlayerTypeLoader'
        );
    });

    QUnit.test('a mount point is added to root', function(assert) {
        uiLoader(
            this.video,
            this.root,
            this.store,
            this.clipGenerator,
            DEFAULT_OPTS
        );

        const mountPoint = this.root.getElementsByClassName('player-ui');
        assert.equal(mountPoint.length, 1, 'only one mountpoint');
    });
});
