import {
    MONTH,
    NORMAL_YEAR,
    LEAP_YEAR,
    assertIsDate,
    clearTime,
    cloneDate,
    getMaxDays,
    isLeapYear,
    isMinimumAge,
} from 'util/date';
import each from 'lodash/each';
import sinon from 'sinon';

const sandbox = sinon.sandbox.create();

// since we can't spy on an imported function
function testDateAssertion(assert, fn) {
    [
        Date.now(),
        '',
        null,
        undefined,
        [],
        {},
        true,
    ].forEach(value => {
        assert.throws(() => {
            fn(value);
        }, Error, `should throw an error when for ${JSON.stringify(value)}`);
    });

    try {
        fn(new Date());
        assert.ok(true, 'should not throw an exception');
    } catch (e) {
        assert.ok(false, `should not throw an exception: ${e}`);
    }
}

QUnit.module('util | date', function() {
    QUnit.test('months as a constant', function(assert) {
        const ALL_MONTHS = [
            MONTH.JANUARY,
            MONTH.FEBRUARY,
            MONTH.MARCH,
            MONTH.APRIL,
            MONTH.MAY,
            MONTH.JUNE,
            MONTH.JULY,
            MONTH.AUGUST,
            MONTH.SEPTEMBER,
            MONTH.OCTOBER,
            MONTH.NOVEMBER,
            MONTH.DECEMBER,
        ];
        ALL_MONTHS.forEach((month, index) => {
            assert.strictEqual(month, index, `should be a 0th based month for ${month} => ${index}`);
        });
        assert.throws(() => {
            MONTH.foo = 12;
        }, TypeError, 'should be immutable');
    });

    QUnit.test('normal year: number of days in a month', function(assert) {
        const expected = Object.freeze({
            [MONTH.JANUARY]: 31,
            [MONTH.FEBRUARY]: 28,
            [MONTH.MARCH]: 31,
            [MONTH.APRIL]: 30,
            [MONTH.MAY]: 31,
            [MONTH.JUNE]: 30,
            [MONTH.JULY]: 31,
            [MONTH.AUGUST]: 31,
            [MONTH.SEPTEMBER]: 30,
            [MONTH.OCTOBER]: 31,
            [MONTH.NOVEMBER]: 30,
            [MONTH.DECEMBER]: 31,
        });
        assert.deepEqual(NORMAL_YEAR, expected, 'should have the correct number of days per month');
        assert.throws(() => {
            NORMAL_YEAR[MONTH.FEBRUARY] = 20;
        }, TypeError, 'should be immutable');
    });

    QUnit.test('leap year: number of days in a month', function(assert) {
        const expected = Object.freeze({
            [MONTH.JANUARY]: 31,
            [MONTH.FEBRUARY]: 29,
            [MONTH.MARCH]: 31,
            [MONTH.APRIL]: 30,
            [MONTH.MAY]: 31,
            [MONTH.JUNE]: 30,
            [MONTH.JULY]: 31,
            [MONTH.AUGUST]: 31,
            [MONTH.SEPTEMBER]: 30,
            [MONTH.OCTOBER]: 31,
            [MONTH.NOVEMBER]: 30,
            [MONTH.DECEMBER]: 31,
        });
        assert.deepEqual(LEAP_YEAR, expected, 'should have the correct number of days per month');
        assert.throws(() => {
            LEAP_YEAR[MONTH.JANUARY] = 12;
        }, TypeError, 'should be immutable');
    });

    QUnit.test('assertIsDate', function(assert) {
        testDateAssertion(assert, assertIsDate);
    });

    QUnit.test('clearTime', function(assert) {
        const date = new Date(2017, 1, 2, 3, 4, 5);
        const actual = clearTime(date);
        const expected = new Date(2017, 1, 2, 0, 0, 0);

        assert.equal(actual.getTime(), expected.getTime(), 'should set the hours to midnight');
        assert.equal(
            date.getTime(),
            new Date(2017, 1, 2, 3, 4, 5).getTime(),
            'should not mutate the original date'
        );
        testDateAssertion(assert, clearTime);
    });

    QUnit.test('cloneDate', function(assert) {
        const date = new Date(2017, 1, 2, 3, 4, 5);
        const cloned = cloneDate(date);

        cloned.setHours(10, 0, 0, 0);
        assert.equal(
            date.getTime(),
            new Date(2017, 1, 2, 3, 4, 5).getTime(),
            'should not mutate the original date'
        );
        testDateAssertion(assert, cloneDate);
    });

    QUnit.test('getMaxDays', function(assert) {
        [
            -1,
            13,
            null,
            undefined,
            [],
            {},
            '',
            '0',
            true,
        ].forEach(value => {
            assert.throws(() => {
                getMaxDays(value, 2004);
            }, Error, `should throw an exception when the value is '${JSON.stringify(value)}'`);
        });

        // sanity check, both are covered in normal/leap year tests
        assert.equal(getMaxDays(MONTH.FEBRUARY, 2004), 29, 'should return 29 days for 2004-02');
        assert.equal(getMaxDays(MONTH.FEBRUARY, 2005), 28, 'should return 28 days for 2005-02');
    });

    QUnit.test('isLeapYear', function(assert) {
        const years = Object.freeze({
            1752: true,
            1753: false,
            1900: false,
            1980: true,
            1981: false,
            1982: false,
            1983: false,
            1984: true,
            1988: true,
            1992: true,
            1994: false,
            1996: true,
            2000: true,
            2004: true,
            2007: false,
            2008: true,
            2017: false,
            2020: true,
        });
        each(years, (expected, year) => {
            const actual = isLeapYear(Number(year));
            assert.equal(actual, expected, `should be ${expected} for ${year}`);
        });

        [
            -1,
            NaN,
            Infinity,
            -Infinity,
            '2004',
            [],
            {},
            '',
            null,
            false,
            2004.120,
        ].forEach(year => {
            assert.throws(() => {
                isLeapYear(year);
            }, Error, `should throw an exception when the year is invalid (year=${year})`);
        });
    });

    QUnit.module('isMinimumAge', function(hooks) {
        hooks.beforeEach(function() {
            sandbox.
                stub(Date, 'now').
                returns(new Date(2017, MONTH.OCTOBER, 29, 12, 29, 59).getTime());
        });

        hooks.afterEach(function() {
            sandbox.restore();
        });

        QUnit.test('date: errors/exceptions', function(assert) {
            [
                Date.now(),
                '',
                null,
                undefined,
                [],
                {},
                true,
            ].forEach(value => {
                assert.throws(() => {
                    isMinimumAge(value, 0);
                }, Error, `should throw an error when for ${JSON.stringify(value)}`);
            });
        });

        QUnit.test('age: errors/exceptions', function(assert) {
            [
                -1,
                NaN,
                Infinity,
                -Infinity,
                '21',
                false,
                21.22,
            ].forEach(age => {
                assert.throws(() => {
                    isMinimumAge(new Date(), age);
                }, Error, `should throw an error when the age is '${age}'}`);
            });
        });

        QUnit.test('minimum age: 19', function(assert) {
            function testMinimumAge(age, expected) {
                const inputDate = new Date(2017 - age, MONTH.OCTOBER, 29, 12, 29, 59);
                assert.strictEqual(
                    isMinimumAge(inputDate, 19),
                    expected,
                    `should be '${expected}' when the age is ${age}`
                );
            }

            testMinimumAge(17, false);
            testMinimumAge(18, false);
            testMinimumAge(19, true);
            testMinimumAge(20, true);
            testMinimumAge(21, true);
            testMinimumAge(22, true);
        });
    });
});
