import { toHumanReadable } from 'util/duration-utils';

QUnit.module('util | toHumanReadable', function(hooks) {
    hooks.beforeEach(function() {
        this.oneSec = 1000;
        this.oneMin = this.oneSec * 60;
        this.oneHour = this.oneMin * 60;
    });

    QUnit.test('0 is valid', function(assert) {
        assert.deepEqual(toHumanReadable(0), {
            hours: 0,
            minutes: 0,
            seconds: 0,
        }, '0 is valid input and all counter values are 0');
    });

    QUnit.test('Converts hours properly and rounds down to integer', function(assert) {
        assert.equal(toHumanReadable(this.oneHour - 1).hours, 0, 'returns 0 for less than 1 hour');
        assert.equal(toHumanReadable(this.oneHour).hours, 1, 'converts 1 hour properly');
        assert.equal(toHumanReadable(2 * this.oneHour).hours, 2, 'converts multiple hours properly');
        assert.equal(toHumanReadable(this.oneHour + (59 * this.oneMin)).hours, 1, 'rounds hours down to whole integer');
    });

    QUnit.test('Converts minutes properly and rounds down to integer', function(assert) {
        assert.equal(toHumanReadable(this.oneMin - 1).hours, 0, 'returns 0 for less than 1 min');
        assert.equal(toHumanReadable(this.oneMin).minutes, 1, 'converts 1 min properly');
        assert.equal(toHumanReadable(2 * this.oneMin).minutes, 2, 'converts multiple mins properly');
        assert.equal(toHumanReadable(this.oneMin + (59 * this.oneSec)).minutes, 1, 'rounds mins down to whole integer');
    });

    QUnit.test('Converts seconds properly and rounds down to integer', function(assert) {
        assert.equal(toHumanReadable(this.oneSec - 1).hours, 0, 'returns 0 for less than 1 second');
        assert.equal(toHumanReadable(this.oneSec).seconds, 1, 'converts 1 sec properly');
        assert.equal(toHumanReadable(2 * this.oneSec).seconds, 2, 'converts multiple secs properly');
        assert.equal(toHumanReadable(this.oneSec + 999).seconds, 1, 'rounds secs down to whole integer');
    });

    QUnit.test('Converts mixed values properly', function(assert) {
        const oneHourOneMinOneSec = this.oneHour + this.oneMin + this.oneSec;
        assert.deepEqual(toHumanReadable(oneHourOneMinOneSec), {
            hours: 1,
            minutes: 1,
            seconds: 1,
        }, '1:1:1 calculated correctly');
        const oneHourZeroMinOneSec = this.oneHour + this.oneSec;
        assert.deepEqual(toHumanReadable(oneHourZeroMinOneSec), {
            hours: 1,
            minutes: 0,
            seconds: 1,
        }, '1:0:1 calculated correctly');
        const zeroHourOneMinOneSec = this.oneMin + this.oneSec;
        assert.deepEqual(toHumanReadable(zeroHourOneMinOneSec), {
            hours: 0,
            minutes: 1,
            seconds: 1,
        }, '0:1:1 calculated correctly');
        const beforeOneDay = (23 * this.oneHour) + (59 * this.oneMin) + (59 * this.oneSec) + 999;
        assert.deepEqual(toHumanReadable(beforeOneDay), {
            hours: 23,
            minutes: 59,
            seconds: 59,
        }, '23:59:59 (residual 999 ms) calculated correctly');
    });
});
