import { FullScreen } from 'util/fullscreen';
import { unitTest } from 'tests/utils/module';
import { init as initStore } from 'state';
import { setFullScreen } from 'actions/screen-mode';
import { ACTION_EMIT_FULLSCREEN_CHANGE } from 'actions/event-emitter';
import BigScreen from 'bigscreen/bigscreen';
import sinon from 'sinon';

unitTest('fullscreen', function(hooks) {
    hooks.beforeEach(function() {
        this.store = initStore();
        this.getScreenModeState = () => {
            return this.store.getState().screenMode;
        };
    });

    hooks.afterEach(function() {
        BigScreen._reset();
    });

    QUnit.test('sets data-isfullscreen _onFullScreenChange', function(assert) {
        const domElement = document.createElement('div');
        new FullScreen(domElement, this.store);

        this.store.dispatch(setFullScreen(true));

        assert.equal(domElement.dataset.isfullscreen, 'true', 'sets isfullscreen true');

        this.store.dispatch(setFullScreen(false));

        assert.equal(domElement.dataset.isfullscreen, 'false', 'sets isfullscreen false');
    });

    QUnit.test('should set canFullScreen in store on construction', function(assert) {
        const domElement = document.createElement('div');

        const beforeConstuction = this.getScreenModeState().canFullScreen;
        assert.equal(beforeConstuction, false, 'init canFullScreen to false');

        // eslint-disable-next-line no-unused-vars
        const fullscreen = new FullScreen(domElement, this.store);

        const afterConstruction = this.getScreenModeState().canFullScreen;
        assert.equal(afterConstruction, true, 'canFullScreen should be set to true');
    });

    QUnit.test('_checkWebkitFullscreen should return false if no video tag found', function(assert) {
        const domElement = document.createElement('div');
        const fullscreen = new FullScreen(domElement, this.store);
        assert.equal(
            fullscreen._checkWebkitFullscreen(),
            false,
            'false if no video tag found'
        );
    });

    // eslint-disable-next-line max-len
    QUnit.test('_checkWebkitFullscreen should check video tags webkitSupportsFullscreen', function(assert) {
        const domElement = document.createElement('div');
        domElement.innerHTML = '<video></video>';
        const fullscreen = new FullScreen(domElement, this.store);
        assert.equal(
            fullscreen._checkWebkitFullscreen(),
            true,
            'true if webkitSupportsFullscreen is on video tag'
        );
    });

    // eslint-disable-next-line max-len
    QUnit.test('_checkWebkitFullscreen returns false if webkitSupportsFullscreen is false', function(assert) {
        const domElement = document.createElement('div');
        const videoTag = document.createElement('video').style.display = 'none';
        domElement.innerHTML = videoTag;
        const fullscreen = new FullScreen(domElement, this.store);
        assert.equal(
            fullscreen._checkWebkitFullscreen(),
            false,
            'false if element doesn\'t support webkitSupportsFullscreen'
        );
    });

    QUnit.test('dispatching setFullScreen should trigger fullscreenchange event to webclient', function(assert) {
        const domElement = document.createElement('div');
        new FullScreen(domElement, this.store);
        sinon.spy(this.store, 'dispatch');
        this.store.dispatch(setFullScreen(true));

        assert.deepEqual(this.store.dispatch.secondCall.args[0], {
            type: ACTION_EMIT_FULLSCREEN_CHANGE,
        }, 'fullscreenchange event fired from middleware');
    });

    QUnit.test('fires _onFullScreenChange on document fullscreenchange', function(assert) {
        const domElement = document.createElement('div');
        new FullScreen(domElement, this.store);
        sinon.spy(this.store, 'dispatch');

        const fullscreenChangeEvent = new CustomEvent('fullscreenchange');
        document.dispatchEvent(fullscreenChangeEvent);

        assert.deepEqual(this.store.dispatch.firstCall.args[0], {
            type: ACTION_EMIT_FULLSCREEN_CHANGE,
        });

        assert.deepEqual(
            this.store.dispatch.secondCall.args[0],
            setFullScreen(false),
            '_onFullScreenChange responded to document\'s fullscreenchange event'
        );
    });
});
