import { reactTest } from 'tests/utils/react-test';
import sinon from 'sinon';
import { popoutPlayer } from 'util/popout-player';
import { playerHost, popoutSize } from 'settings';
import * as Params from 'util/params';
import * as Timestamp from 'util/timestamp';
import { LiveTwitchContentStream } from 'stream/twitch-live';
import { VODTwitchContentStream } from 'stream/twitch-vod';
import { PLAYER_SITE, PLAYER_IMDB } from 'util/player-type';

reactTest('util | popout-player', function(hooks) {
    hooks.beforeEach(function() {
        this.fakeCollection = {
            id: 'fakecollectionid',
        };

        this.fakePlayback = {
            volume: 100,
            currentTime: 12345,
            muted: false,
        };

        this.fakeWindow = {
            open: sinon.spy(),
        };

        // eslint-disable-next-line max-len
        this.expectedArgs = `width=${popoutSize.width},height=${popoutSize.height},toolbar=no,menubar=no,scrollbars=no,location=no,status=no`;
    });

    QUnit.test('for a channel stream, constructs expected url and opens it in window', function(assert) {
        const fakeChannel = 'monstercat';
        const fakeLiveStream = new LiveTwitchContentStream(fakeChannel, Promise.resolve(''), {}, {}, {});
        popoutPlayer({
            collection: this.fakeCollection,
            playback: this.fakePlayback,
            stream: fakeLiveStream,
            window: this.fakeWindow,
            playerType: PLAYER_SITE,
        });

        const expectedOptions = {
            volume: this.fakePlayback.volume,
            muted: this.fakePlayback.muted,
            channel: fakeChannel,
        };
        const expectedUrl = `${playerHost}/?${Params.toString(expectedOptions)}`;

        assert.equal(this.fakeWindow.open.callCount, 1, 'one new window opened');
        assert.equal(this.fakeWindow.open.firstCall.args[0], expectedUrl, 'url is as expected');
        assert.equal(this.fakeWindow.open.firstCall.args[1], '_blank', 'opened in _blank session');
        assert.equal(this.fakeWindow.open.firstCall.args[2], this.expectedArgs, 'has expected args');
    });

    QUnit.test('for a vod stream, constructs expected url and opens it in window', function(assert) {
        const fakeVodId = '1111';
        const fakeVODStream = new VODTwitchContentStream(fakeVodId, Promise.resolve(''), {}, {}, {});
        popoutPlayer({
            collection: this.fakeCollection,
            playback: this.fakePlayback,
            stream: fakeVODStream,
            window: this.fakeWindow,
            playerType: PLAYER_SITE,
        });

        const expectedOptions = {
            volume: this.fakePlayback.volume,
            muted: this.fakePlayback.muted,
            video: fakeVodId,
            collection: this.fakeCollection.id,
            time: Timestamp.toURLString(this.fakePlayback.currentTime),
        };

        const expectedUrl = `${playerHost}/?${Params.toString(expectedOptions)}`;
        assert.equal(this.fakeWindow.open.callCount, 1, 'one new window opened');
        assert.equal(this.fakeWindow.open.firstCall.args[0], expectedUrl, 'url is as expected');
        assert.equal(this.fakeWindow.open.firstCall.args[1], '_blank', 'opened in _blank session');
        assert.equal(this.fakeWindow.open.firstCall.args[2], this.expectedArgs, 'has expected args');
    });

    QUnit.test('for imdb playerType, adds &player=imdb to list of params', function(assert) {
        const fakeVodId = '1111';
        const fakeVODStream = new VODTwitchContentStream(fakeVodId, Promise.resolve(''), {}, {}, {});
        popoutPlayer({
            collection: this.fakeCollection,
            playback: this.fakePlayback,
            stream: fakeVODStream,
            window: this.fakeWindow,
            playerType: PLAYER_IMDB,
        });

        const expectedOptions = {
            volume: this.fakePlayback.volume,
            muted: this.fakePlayback.muted,
            video: fakeVodId,
            collection: this.fakeCollection.id,
            time: Timestamp.toURLString(this.fakePlayback.currentTime),
            player: PLAYER_IMDB,
        };

        const expectedUrl = `${playerHost}/?${Params.toString(expectedOptions)}`;
        assert.equal(this.fakeWindow.open.callCount, 1, 'one new window opened');
        assert.equal(this.fakeWindow.open.firstCall.args[0], expectedUrl, 'url is as expected');
        assert.equal(this.fakeWindow.open.firstCall.args[1], '_blank', 'opened in _blank session');
        assert.equal(this.fakeWindow.open.firstCall.args[2], this.expectedArgs, 'has expected args');
    });
});
